<?php

namespace xCloud\MigrationAssistant;

use Exception;
use WP_Error;
use WP_REST_Request;

class Rest
{
    public function __construct()
    {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    function register_rest($patth, $callback)
    {
        register_rest_route('xcloud-migration/v1', $patth, [
            'methods' => 'GET',
            'callback' => function (WP_REST_Request $request) use ($callback) {
                return $callback($request);
            },
            'permission_callback' => function (WP_REST_Request $request) {
                return current_user_can('manage_options');
            }
        ]);
    }

    function register_secure_rest($patth, $callback)
    {
        register_rest_route('xcloud-migration/v1', $patth, [
            'methods' => ['GET', 'POST'],
            'callback' => function (WP_REST_Request $request) use ($callback) {
                $params = @json_decode(base64_decode($request->get_param('payload')), true);

                if (!$params) {
                    $params = [];
                }

                $response = $callback($params);

                if (is_wp_error($response) || (defined('XCLOUD_NO_ENCRYPTION') && XCLOUD_NO_ENCRYPTION)) {
                    return $response;
                }

                return Response::withEncryption($response);
            },
            'permission_callback' => function (WP_REST_Request $request) {
                $auth_token = xcloud_migration()->getOption('auth_token');
                return $auth_token && ($request->get_param('auth_token') == $auth_token);
            },
        ]);
    }

    function register_routes()
    {
        $this->register_secure_rest('/site_info', [$this, 'get_site_info']);
        $this->register_secure_rest('/abspath', [$this, 'get_abspath']);
        $this->register_secure_rest('/wp_config', [$this, 'get_wp_config']);
        $this->register_secure_rest('/db/tables', [$this, 'get_db_tables']);
        $this->register_secure_rest('/db/table_structure', [$this, 'get_db_table_structure']);
        $this->register_secure_rest('/db/table_data', [$this, 'get_table_data']);
        $this->register_secure_rest('/db/mysqldump', [$this, 'mysqldump']);
        $this->register_secure_rest('/fs/structure', [$this, 'get_file_system_structure']);
        $this->register_secure_rest('/fs/hash', [$this, 'hash']);
        $this->register_secure_rest('/fs/size', [$this, 'size']);
        $this->register_secure_rest('/fs/compress', [$this, 'compress']);
        $this->register_secure_rest('/fs/delete', [$this, 'delete']);
        $this->register_secure_rest('/fs/read', [$this, 'get_file_system_server']);
        $this->register_secure_rest('/server/memory_limit', [$this, 'get_server_memory_limit']);
        $this->register_secure_rest('/receive_processing_statuses', [$this, 'receive_migration_statuses']);
        $this->register_rest('/get_statuses', [$this, 'get_status']);
    }

    function get_abspath($params)
    {
        return [
            'abspath'         => ABSPATH,
            'plugin_version'  => xcloud_migration()->getVersion(),
        ];
    }

    function get_site_info($params)
    {
        return [
            'mysqldump'       => DatabaseMigration::check_mysqldump_availability(),
            'memory_limit'    => xcloud_migration_get_server_memory_limit(),
            'time_limit'      => ini_get( 'max_execution_time' ),
            'database_size'   => $this->get_database_size(),
            'disk_free_space' => function_exists( 'disk_free_space' ) ? @disk_free_space(ABSPATH) : null,
        ];
    }

    public function get_database_size() {
        global $wpdb;

        $tables = $wpdb->get_results('SHOW TABLE STATUS');

        $size           = 0;
        $row_count      = 0;
        $avg_row_length = 0;

        foreach ($tables as $table) {
            $size           += $table->Data_length + $table->Index_length;
            $row_count      += $wpdb->get_var("SELECT COUNT(*) FROM {$table->Name}");
            $avg_row_length  = max($table->Avg_row_length, $avg_row_length);
        }

        // Convert size to MB
        $size = $size / 1024 / 1024;

        return [
            'size'           => $size,
            'row_count'      => $row_count,
            'table_count'    => count($tables),
            'avg_row_length' => $avg_row_length,
        ];
    }

    function get_wp_config($params)
    {
        return (new FileSystemMigration)->getConfig();
    }

    function get_db_tables($params)
    {
        return (new DatabaseMigration)->getTables();
    }

    function get_db_table_structure($params)
    {
        $table = isset($params['table']) ? $params['table'] : null;

        if (!$table) {
            return new WP_Error('no_table', 'No table specified');
        }

        return (new DatabaseMigration)->getTableStructure($table);
    }

    function get_table_data($params)
    {
        $table = isset($params['table']) ? $params['table'] : null;
        $row_start = (int) isset($params['row_start']) && $params['row_start'] ? $params['row_start'] : 0;
        $row_inc = (int) isset($params['row_inc']) && $params['row_inc'] ? $params['row_inc'] : 10;

        if (!$table) {
            return new WP_Error('no_table', 'No table specified');
        }

        return (new DatabaseMigration)->getTableData($table, $row_start, $row_inc);
    }

    function get_file_system_structure($params)
    {
        $dir = isset($params['root']) && $params['root'] ? $params['root'] : ABSPATH;
        $recursive = (bool) isset($params['recursive']) ? $params['recursive'] : false;

        return (new FileSystemMigration)->getStructure($dir, $recursive);
    }

    function compress($params)
    {
        $dir = isset($params['dir']) && $params['dir'] ? $params['dir'] : null;
        $zip_path = isset($params['zip_path']) && $params['zip_path'] ? $params['zip_path'] : null;
        $clean_folder = isset($params['clean_folder']) && $params['clean_folder'] ? $params['clean_folder'] : false;

        if (!$dir) {
            return new WP_Error('no_dir', 'No dir specified');
        }

        if (!$zip_path) {
            return new WP_Error('no_zip_path', 'No zip path specified');
        }

        $folder_size = (new FileCompressor())->getFolderSize($dir);

        if ($folder_size > xcloud_migration_get_server_memory_limit()) {
            return new WP_Error('memory_limit', 'Memory limit exceeded');
        }

        $status = (new FileCompressor())->zipDirectory($dir, $zip_path, $clean_folder);

        if (is_wp_error($status)) {
            return $status;
        }

        if (!$status) {
            return new WP_Error('zip_error', 'Failed to create zip file');
        }

        $zip_size = filesize($zip_path);

        if ($zip_size === false) {
            return new WP_Error('zip_error', 'Failed to create zip file');
        }

        return [
            'folder_size' => $folder_size,
            'size' => $zip_size,
            'hash' => hash_file('sha256', $zip_path),
            'path' => $zip_path
        ];
    }

    function hash($params)
    {
        $path = isset($params['path']) && $params['path'] ? $params['path'] : null;

        if (!$path) {
            return new WP_Error('no_path', 'No path specified');
        }

        return [
            'hash' => hash_file('sha256', $path),
            'path' => $path
        ];
    }

    function size($params)
    {
        $path = isset($params['path']) && $params['path'] ? $params['path'] : null;
        $include_count = isset($params['include_count']) && $params['include_count'] ? $params['include_count'] : null;

        if (!$path) {
            return new WP_Error('no_path', 'No path specified');
        }

        if (is_dir($path) && $include_count) {
            list($size, $count) = (new FileCompressor())->getFolderSizeAndCount($path);
            return [
                'type' => 'dir',
                'size' => $size,
                'count' => $count,
                'path' => $path
            ];
        } elseif (is_dir($path)) {
            return [
                'type' => 'dir',
                'size' => (new FileCompressor())->getFolderSize($path),
                'path' => $path
            ];
        } elseif (is_file($path)) {
            return [
                'type' => 'file',
                'size' => filesize($path),
                'path' => $path
            ];
        }

        return new WP_Error('invalid_path', 'Invalid path specified');
    }

    function delete($params)
    {
        $path = isset($params['path']) && $params['path'] ? $params['path'] : null;

        if (!$path) {
            return new WP_Error('path_error', 'No path specified');
        }

        return (new FileSystemMigration)->deleteFile($path);
    }

    function get_file_system_server($params)
    {
        $start = (int) isset($params['start']) ? $params['start'] : 0;
        $file = isset($params['file']) ? $params['file'] : '';
        $files = isset($params['files']) ? $params['files'] : [];

        if ($file) {
            return (new FileSystemMigration)->getFile($file, $start);
        }

        if ($files) {
            return (new FileSystemMigration)->getFiles($files);
        }

        return new WP_Error('no_file', 'No file specified');
    }

    function get_server_memory_limit()
    {
        return [
            'memory_limit' => xcloud_migration_get_server_memory_limit()
        ];
    }

    function receive_migration_statuses($params)
    {
        return [
            'data' => $params,
            'saved' => xCloudOption::set('migration', array_merge(
                    $params['statuses'],
                    [
                        'migrating_from' => $params['migrating_from'],
                        'migrating_to' => $params['migrating_to'],
                        'state' => $params['state'],
                        'last_updated' => time()
                    ]
                )
            )
        ];
    }

    function get_status(WP_REST_Request $request)
    {
        return array_merge(
            ['data' => xCloudOption::get('migration')],
            xCloudOption::requiredDataForProgress()
        );
    }

    public function mysqldump($params)
    {
        $result = [];

        wp_mkdir_p(dirname($params['dump_path']));

        switch ($params['method']) {
            case 'php':
                $result = (new DatabaseMigration)->_mysqldump($params);
                if(file_exists($result['dumpPath'])){
                    $result['sql_hash'] = hash_file('sha256', $result['dumpPath']);
                }
                break;
            case 'shell':
                $result = (new DatabaseMigration)->mysqldump($params);
                if(file_exists($result['dumpPath'])){
                    $result['sql_hash'] = hash_file('sha256', $result['dumpPath']);
                }
                break;
            default:
                return new WP_Error('invalid_method', 'Invalid method specified');
        }

        return $result;
    }

}