<?php

namespace xCloud\MigrationAssistant;

use Exception;
use WP_Error;

class DatabaseMigration
{
    protected $file_size = 0;

    function getTables()
    {
        global $wpdb;

        $list_tables = $wpdb->get_results("SHOW TABLES");

        $tables = [];

        foreach ($list_tables as $mytable) {
            foreach ($mytable as $t) {
                $tables[] = $t;
            }
        }

        return [
            'tables' => $tables,
        ];
    }

    function getTableStructure($table)
    {
        global $wpdb;

        if (!$this->tableExists($table)) {
            return new WP_Error('table_not_found', 'Table not found');
        }

        $createTable = $wpdb->get_results("SHOW CREATE TABLE $table", ARRAY_N);
        $describeTable = $wpdb->get_results("DESCRIBE TABLE $table");
        $rowCount = $wpdb->get_var("SELECT COUNT(*) FROM $table");

        if (false === $createTable || !isset($createTable[0][1])) {
            return new WP_Error('error_show_create_table',
                sprintf(__('Error with SHOW CREATE TABLE for %s.', 'wp-xcloud-migration'), $table)
            );
        }

        if (!isset($createTable[0][1])) {
            return new WP_Error('error_describe_table',
                sprintf(__('Error with DESCRIBE TABLE for %s.', 'wp-xcloud-migration'), $table)
            );
        }

        return [
            'table' => $createTable[0][0],
            'create_table' => $createTable[0][1],
            'describe_table' => isset($describeTable[0]) ? $describeTable[0] : null,
            'row_count' => $rowCount,
        ];
    }

    function getTableData($table, $rowStart, $rowIncrement)
    {
        global $wpdb;

        if (!$this->tableExists($table)) {
            return new WP_Error('table_not_found', 'Table not found');
        }

        $where = '';

        $data = $wpdb->get_results("SELECT * FROM $table $where LIMIT {$rowStart}, {$rowIncrement}", ARRAY_A);

        return [
            'table' => $table,
            'data' => $data,
            'next' => $rowStart + $rowIncrement,
        ];
    }

    /**
     * @param $table
     * @return string|null
     */
    private function tableExists($table)
    {
        global $wpdb;

        return $wpdb->get_var(
            $wpdb->prepare(
                "SELECT count(*) FROM information_schema.tables WHERE table_schema='%s' AND table_name='%s' LIMIT 1;",
                DB_NAME,
                $table
            )
        );
    }

    /**
     * Create a MySQL dump of the database
     *
     * @param string $outputFilePath
     * @return string|WP_Error
     */
    public function createMysqlDump($outputFilePath) {
        $hostDetails = explode(':', DB_HOST);
        $databaseHost = $hostDetails[0];
        $databasePort = isset($hostDetails[1]) ? $hostDetails[1] : '3306'; // Default to 3306 if no port is set

        $databaseUsername = DB_USER;
        $databasePassword = DB_PASSWORD;
        $databaseName = DB_NAME;

        $dumpCommand = "mysqldump --no-tablespaces --host={$databaseHost} --port={$databasePort} --user={$databaseUsername} --password={$databasePassword} {$databaseName} > {$outputFilePath} 2>/dev/null";
        shell_exec($dumpCommand);

        if (!file_exists($outputFilePath) || filesize($outputFilePath) <= 0) {
            return new WP_Error('dump_failed', 'Failed to dump database');
        }

        return $outputFilePath;
    }

    public function createChunk($sourceFilePath, $startPosition, $maxFileSize) {
        $chunkSize     = 1 * MB_IN_BYTES; // Size of each chunk in bytes
        $chunkFilePath = wp_tempnam(basename($sourceFilePath)) . "_chunk";
        $fileHandle    = fopen($sourceFilePath, "r");
        $outputHandle  = fopen($chunkFilePath, "w");

        fseek($fileHandle, $startPosition);
        $bytesRead = 0;
        while ($bytesRead < $maxFileSize * MB_IN_BYTES && !feof($fileHandle)) {
            $chunkData = fread($fileHandle, $chunkSize);
            fwrite($outputHandle, $chunkData);
            $bytesRead += strlen($chunkData);
        }

        fclose($fileHandle);
        fclose($outputHandle);

        $nextStartPosition = $startPosition + $bytesRead;
        if ($nextStartPosition >= filesize($sourceFilePath)) {
            $nextStartPosition = null; // No more chunks left
        }

        return [$chunkFilePath, $nextStartPosition];
    }

    public function mysqldump($parameters) {
        if(!function_exists('wp_tempnam')){
            require_once(ABSPATH . 'wp-admin/includes/file.php');
        }

        if(isset($parameters['dumpFilePath']) && file_exists($parameters['dumpFilePath'])){
            $dumpFilePath = $parameters['dumpFilePath'];
        }
        else{
            $dumpFilePath  = wp_tempnam(basename($parameters['dump_path']));
        }
        $maxFileSize   = $parameters['max_file_size'];
        $startPosition = $parameters['start'];

        if ($startPosition == 0) {
            $dumpFilePath = $this->createMysqlDump($dumpFilePath);
            if (is_wp_error($dumpFilePath)) {
                return [
                    'dumpPath' => null,
                    'status'   => 'error',
                    'message'  => $dumpFilePath->get_error_message(),
                ];
            }
        }

        list($chunkFilePath, $nextStartPosition) = $this->createChunk($dumpFilePath, $startPosition, $maxFileSize);

        $zipFilePath = $this->create_zip($chunkFilePath, $parameters['dump_path'] . '.zip' );

        return [
            'nextTable'    => null,
            'nextStart'    => $nextStartPosition,
            'sql_hash'     => null,
            'dumpPath'     => $zipFilePath,
            'dumpFilePath' => $dumpFilePath,
            'dumpSize'     => filesize($dumpFilePath),
            'status'       => $nextStartPosition === null ? 'completed' : 'continue',
        ];
    }


    public static function check_mysqldump_availability() {
        if (!function_exists('shell_exec')) {
            return false;
        }

        $output = shell_exec('which mysqldump');
        if (empty($output)) {
            return false;
        }

        return true;
    }

    public function _mysqldump($params)
    {
        // ini increase execution time
        // ini_set('max_execution_time', 300);
        error_log("\$params" . print_r($params, true) . PHP_EOL);

        $tables = $this->getTables()['tables'];

        $totalRowCount  = 0;
        $dumpedRowCount = 0;

        $dump_path    = $params['dump_path'];
        $rowIncrement = isset($params['rowIncrement']) ? $params['rowIncrement'] : 50;
        $max_file_size = isset($params['max_file_size']) ? $params['max_file_size'] : 50;

        // Get currentTable and currentRow from $params
        $currentTable = isset($params['currentTable']) ? $params['currentTable'] : null;
        $currentRow   = isset($params['currentRow']) ? $params['currentRow'] : 0;

        if (file_exists($dump_path)) {
            unlink($dump_path);
        }

        $tableData = [];

        $startTable = $currentTable ? array_search($currentTable, $tables) : 0;

        for ($i = $startTable; $i < count($tables); $i++) {
            $table = $tables[$i];
            $tableStructure = $this->getTableStructure($table);
            $tableData[$table] = $tableStructure;
            $totalRowCount += $tableStructure['row_count'];
        }


        $this->preDumpText($dump_path);

        foreach ($tableData as $table => $tableStructure) {

            $defs = array();
            $ints = array();
            $start = $table === $currentTable ? $currentRow : 0;

            // echo PHP_EOL."migrating_table $table : row_count: {$tableStructure['row_count']}".PHP_EOL;

            $this->write_create_table($dump_path, $start, $table, $tableStructure);

            while (($tableStructure['row_count'] > 0) && ($start < $tableStructure['row_count'])) {

                $table_data = $this->getTableData($table, $start, $rowIncrement);

                $entries = 'INSERT INTO '.$this->backquote($table).' VALUES (';

                $search = array("\x00", "\x0a", "\x0d", "\x1a");
                $replace = array('\0', '\n', '\r', '\Z');

                foreach ($table_data['data'] as $row) {
                    $values = array();
                    foreach ($row as $key => $value) {
                        if (!empty($ints[strtolower($key)])) {
                            // make sure there are no blank spots in the insert syntax,
                            // yet try to avoid quotation marks around integers
                            $value = (null === $value || '' === $value) ? $defs[strtolower($key)] : $value;
                            $values[] = ('' === $value) ? "''" : $value;
                        } else {
                            $values[] = "'".str_replace($search, $replace, $this->sql_addslashes($value))."'";
                        }
                    }

                    $this->wite_dump($dump_path, " \n".$entries.implode(', ', $values).');');

                    $dumpedRowCount++;

                }

                $start = $table_data['next'];

                if (!$start) {
                    break;
                }

                $file_size_mb = $this->wite_dump($dump_path, "\n\n");
                if ($file_size_mb >= $max_file_size) { // 100 MB
                    error_log("file_size_mb: $file_size_mb" . PHP_EOL);
                    $zip_path = $this->create_zip($dump_path);
                    return [
                        'nextTable'      => $table,
                        'nextRow'        => $start,
                        'dumpPath'       => $zip_path,
                        'status'         => 'continue',
                        'dumpedRowCount' => $dumpedRowCount,
                    ];
                }
            }

            $this->wite_dump($dump_path, "/*!40000 ALTER TABLE ".$this->backquote($table)." ENABLE KEYS */;"."\n\n\n");
        }

        $this->afterDumpText($dump_path);

        $zip_path = $this->create_zip($dump_path);
        // Default return value
        return [
            'nextTable'      => null,
            'nextRow'        => null,
            'dumpPath'       => $zip_path,
            'status'         => 'completed',
            'dumpedRowCount' => $dumpedRowCount,
        ];
    }

    protected function write_create_table($dump_path, $start, $table, $tableStructure){
        if($start === 0){
            $this->wite_dump($dump_path, "--
-- Table structure for table `$table`
--\n\n");

            $this->wite_dump($dump_path, "DROP TABLE IF EXISTS ".$this->backquote($table).";\n\n");

            $this->wite_dump($dump_path, "/*!40101 SET @saved_cs_client     = @@character_set_client */;\n");
            $this->wite_dump($dump_path, "/*!50503 SET character_set_client = utf8mb4 */;\n");
            $this->wite_dump($dump_path, $tableStructure['create_table'].";\n");
            $this->wite_dump($dump_path, "/*!40101 SET character_set_client = @saved_cs_client */;\n\n\n");

            $this->wite_dump($dump_path, "/*!40000 ALTER TABLE ".$this->backquote($table)." DISABLE KEYS */;"."\n");
        }
    }

    protected function wite_dump($dump_path, $content)
    {

        $this->file_size += file_put_contents($dump_path, $content, FILE_APPEND);


        // Check file size
        $file_size_mb = $this->file_size / (1024 * 1024); // size in MB

        return $file_size_mb;
    }

    protected function create_zip($dump_path, $zip_path = null){
        if(!$zip_path){
            $zip_path = $dump_path . '.zip';
        }
        $zip = new \ZipArchive();
        if ($zip->open($zip_path, \ZipArchive::CREATE) === TRUE) {
            $zip->addFile($dump_path, basename($dump_path));
            // $auth_token     = xCloudOption::get('settings.auth_token');
            $encryption_key = xCloudOption::get('settings.encryption_key');

            // Set encryption for the file
            $zip->setEncryptionName(basename($dump_path), \ZipArchive::EM_AES_256, $encryption_key);

            $zip->close();
            unlink($dump_path);
            return $zip_path;
        }
        return false;
    }


    /**
     * @param  mixed  $dump_path
     * @return void
     */
    protected function preDumpText(mixed $dump_path): void
    {
        $this->wite_dump($dump_path, "-- MySQL dump via xCloud Migration Assistant
--
-- Host: localhost    Database: wp
-- ------------------------------------------------------
-- Server version	8.0.27

/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;
/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;
/*!40101 SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION */;
/*!50503 SET NAMES utf8mb4 */;
/*!40103 SET @OLD_TIME_ZONE=@@TIME_ZONE */;
/*!40103 SET TIME_ZONE='+00:00' */;
/*!40014 SET @OLD_UNIQUE_CHECKS=@@UNIQUE_CHECKS, UNIQUE_CHECKS=0 */;
/*!40014 SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0 */;
/*!40101 SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO' */;
/*!40111 SET @OLD_SQL_NOTES=@@SQL_NOTES, SQL_NOTES=0 */;\n\n\n
");
    }

    /**
     * @param  mixed  $dump_path
     * @return void
     */
    protected function afterDumpText(mixed $dump_path): void
    {
        $this->wite_dump($dump_path, "/*!40103 SET TIME_ZONE=@OLD_TIME_ZONE */;
/*!40101 SET SQL_MODE=@OLD_SQL_MODE */;
/*!40014 SET FOREIGN_KEY_CHECKS=@OLD_FOREIGN_KEY_CHECKS */;
/*!40014 SET UNIQUE_CHECKS=@OLD_UNIQUE_CHECKS */;
/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;
/*!40101 SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS */;
/*!40101 SET COLLATION_CONNECTION=@OLD_COLLATION_CONNECTION */;
/*!40111 SET SQL_NOTES=@OLD_SQL_NOTES */;\n\n", true);
    }

    /**
     * Add backquotes to tables and db-names in
     * SQL queries. Taken from phpMyAdmin.
     */
    protected function backquote($a_name)
    {
        if (!empty($a_name) && $a_name != '*') {
            if (is_array($a_name)) {
                $result = array();
                reset($a_name);
                foreach ($a_name as $key => $val) {
                    $result[$key] = '`'.$val.'`';
                }
                return $result;
            } else {
                return '`'.$a_name.'`';
            }
        } else {
            return $a_name;
        }
    }

    /**
     * Better addslashes for SQL queries.
     * Taken from phpMyAdmin.
     */
    protected function sql_addslashes($a_string = '', $is_like = false)
    {
        if (is_null($a_string)) {
            return '';
        }

        if ($is_like) {
            $a_string = str_replace('\\', '\\\\\\\\', $a_string);
        } else {
            $a_string = str_replace('\\', '\\\\', $a_string);
        }

        return str_replace('\'', '\\\'', $a_string);
    }
}