<?php

namespace GiveRecurring\PaymentGatewayModules;

use Give\Framework\Exceptions\Primitives\Exception;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionActive;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionCancelled;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionCompleted;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionExpired;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionFailing;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionFirstDonationCompleted;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionRenewalDonationCreated;
use Give\Framework\PaymentGateways\Webhooks\EventHandlers\SubscriptionSuspended;
use Give\Helpers\Hooks;
use Give\PaymentGateways\Gateways\PayPalCommerce\PayPalCommerceGateway;
use Give\PaymentGateways\Gateways\PayPalStandard\PayPalStandard;
use Give\PaymentGateways\Gateways\Stripe\StripePaymentElementGateway\StripePaymentElementGateway;
use Give\PaymentGateways\Gateways\TestGateway\TestGateway;
use Give\PaymentGateways\Gateways\TestOffsiteGateway\TestOffsiteGateway;
use Give\ServiceProviders\ServiceProvider as ServiceProviderInterface;
use Give_Subscription;
use GiveAuthorizeNet\DataTransferObjects\AuthorizeGatewayData;
use GiveAuthorizeNet\Gateway\CreditCardGateway as CreditCardAuthorizeGateway;
use GiveAuthorizeNet\Gateway\eCheckGateway as eCheckAuthorizeGateway;
use GiveBlink\Gateway\BlinkGateway;
use GiveBraintree\Actions\HandleBraintreeGatewayData;
use GiveBraintree\API\BraintreeApiFacade;
use GiveBraintree\Gateway\BraintreeGateway;
use GiveMollie\Gateway\MollieGateway;
use GivePayFast\Gateway\PayFastGateway;
use GiveRazorpay\Actions\HandleRazorpayGatewayData;
use GiveRazorpay\Gateway\RazorpayGateway;
use GiveRecurring\PaymentGatewayModules\Actions\AddPaymentGatewayModulesToLegacyList;
use GiveRecurring\PaymentGatewayModules\Actions\CreateSubscriptionGatewayData;
use GiveRecurring\PaymentGatewayModules\Actions\ModifySubscriptionDonationData;
use GiveRecurring\PaymentGatewayModules\Modules\AuthorizeNet\AuthorizeGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\AuthorizeNet\EventHandlers\HandleSubscriptionDonations;
use GiveRecurring\PaymentGatewayModules\Modules\AuthorizeNet\LegacyListeners\DispatchGiveAuthorizeRecurringPaymentDescription;
use GiveRecurring\PaymentGatewayModules\Modules\AuthorizeNet\Webhooks\SubscriptionWebhooks;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions\CanUpdateBlinkSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\BlinkGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\Webhooks\BlinkPaymentSubscriptionWebhookRequestHandler;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\Actions\CancelBraintreeSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\Actions\CreateBraintreeSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\Admin\Notices;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\BraintreeGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\Webhooks\BraintreeSubscriptionWebhookNotificationHandler;
use GiveRecurring\PaymentGatewayModules\Modules\Mollie\MollieGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Mollie\Webhooks\MollieSubscriptionWebhookRequestHandler;
use GiveRecurring\PaymentGatewayModules\Modules\PayFastGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\PayPalCommerce\PayPalCommerceSubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\PayPalStandard\Adapters\PayPalStandardGatewayLegacyAdapter;
use GiveRecurring\PaymentGatewayModules\Modules\PayPalStandard\Adapters\PayPalStandardGatewayLegacyWebhooksAdapter;
use GiveRecurring\PaymentGatewayModules\Modules\PayPalStandard\PayPalStandardGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Razorpay\RazorpayGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Razorpay\Webhooks\RazorpaySubscriptionWebhookNotificationHandler;
use GiveRecurring\PaymentGatewayModules\Modules\Square\EventHandlers\HandleSquareSubscriptionDonations;
use GiveRecurring\PaymentGatewayModules\Modules\Square\SquareGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Square\Webhooks\SquareWebhookEvents;
use GiveRecurring\PaymentGatewayModules\Modules\Stripe\StripePaymentElementGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Test\TestGatewaySubscriptionModule;
use GiveRecurring\PaymentGatewayModules\Modules\Test\TestOffsiteGatewaySubscriptionModule;
use GiveSquare\GatewayConnect\Modals\SquareAccount;
use GiveSquare\PaymentGateway\Actions\GetSquareGatewayData;
use GiveSquare\PaymentGateway\CreditCardGateway as CreditCardSquareGateway;

class ServiceProvider implements ServiceProviderInterface
{
    /**
     * @since 2.8.0 Register Braintree singleton action classes
     * @since 1.14.0
     *
     * @inheritDoc
     */
    public function register()
    {
        give()->singleton(CreateBraintreeSubscription::class, static function () {
            return new CreateBraintreeSubscription(give(BraintreeApiFacade::class));
        });

        give()->singleton(CancelBraintreeSubscription::class, static function () {
            return new CancelBraintreeSubscription(give(BraintreeApiFacade::class));
        });
    }

    /**
     * @since 2.7.0 Use event handlers classes from Give core
     * @since 2.5.0 add Test Gateway Subscription module
     * @since 2.5.0 add Stripe Payment Element
     * @since 1.14.0
     *
     * @inheritDoc
     * @throws Exception
     * @throws \Exception
     */
    public function boot()
    {
        if (class_exists(PayFastGateway::class)) {
            $payFastGatewayId = PayFastGateway::id();
            add_filter("givewp_gateway_{$payFastGatewayId}_subscription_module", function () {
                return PayFastGatewaySubscriptionModule::class;
            });
        }

        $this->registerTestGatewayModule();
        $this->registerAuthorizeNetModule();
        $this->registerSquareModule();
        $this->registerPayPalStandardModule();
        $this->registerStripePaymentElementModule();
        $this->registerPayPalCommerceModule();
        $this->registerMollieModule();
        $this->registerBraintreeModule();
        $this->registerRazorpayModule();
        $this->registerBlinkPaymentModule();

        Hooks::addFilter('give_recurring_available_gateways', AddPaymentGatewayModulesToLegacyList::class);
    }

    /**
     * @since 2.7.0 Rename "setStatus" to "__invoke"
     * @since 2.3.0
     *
     * @throws Exception
     */
    private function registerSquareModule()
    {
        if (class_exists(CreditCardSquareGateway::class)) {
            $mode = give_is_test_mode() ? 'test' : 'live';
            $squareAccount = SquareAccount::make($mode);

            if ($squareAccount->isConnected() && $squareAccount->validateScopes()) {
                $squareGatewayId = CreditCardSquareGateway::id();
                add_filter("givewp_gateway_{$squareGatewayId}_subscription_module", function () {
                    return SquareGatewaySubscriptionModule::class;
                });

                Hooks::addFilter(
                    sprintf('givewp_create_subscription_gateway_data_%s', $squareGatewayId),
                    GetSquareGatewayData::class
                );

                Hooks::addFilter(
                    sprintf(
                        'givewp_donor_dashboard_edit_subscription_payment_method_gateway_data_%s',
                        $squareGatewayId
                    ),
                    GetSquareGatewayData::class
                );

                Hooks::addAction('givewp_square_webhook_event', SquareWebhookEvents::class, 'processEvent');

                Hooks::addAction(
                    'givewp_square_event_handle_subscription_donations',
                    HandleSquareSubscriptionDonations::class,
                    'run',
                    10,
                    2
                );

                Hooks::addAction(
                    'givewp_square_event_subscription_active',
                    SubscriptionActive::class,
                    '__invoke',
                    10,
                    2
                );

                Hooks::addAction(
                    'givewp_square_event_subscription_cancelled',
                    SubscriptionCancelled::class,
                    '__invoke',
                    10,
                    2
                );

                Hooks::addAction(
                    'givewp_square_event_subscription_suspended',
                    SubscriptionSuspended::class,
                    '__invoke',
                    10,
                    2
                );
            }
        }
    }

    /**
     * @since 2.7.0 Rename "setStatus" to "__invoke"
     * @since 2.3.1 Use givewp_authorize_recurring_payment_description filter
     * @since      2.2.0
     *
     * @throws PaymentGatewayException
     */
    private function registerAuthorizeNetModule()
    {
        if (class_exists(CreditCardAuthorizeGateway::class) && class_exists(eCheckAuthorizeGateway::class)) {
            $creditCardId = CreditCardAuthorizeGateway::id();
            add_filter("givewp_gateway_{$creditCardId}_subscription_module", function () {
                return AuthorizeGatewaySubscriptionModule::class;
            });
            add_filter("givewp_create_subscription_gateway_data_{$creditCardId}", function ($gatewayData) {
                return $this->handleAuthorizeNetGatewayData($gatewayData);
            });
            add_filter(
                "givewp_donor_dashboard_edit_subscription_payment_method_gateway_data_{$creditCardId}",
                function ($gatewayData) {
                    return $this->handleAuthorizeNetGatewayData($gatewayData);
                }
            );

            $eCheckId = eCheckAuthorizeGateway::id();
            add_filter("givewp_gateway_{$eCheckId}_subscription_module", function () {
                return AuthorizeGatewaySubscriptionModule::class;
            });
            add_filter("givewp_create_subscription_gateway_data_{$eCheckId}", function ($gatewayData) {
                return $this->handleAuthorizeNetGatewayData($gatewayData);
            });
            add_filter(
                "givewp_donor_dashboard_edit_subscription_payment_method_gateway_data_{$eCheckId}",
                function ($gatewayData) {
                    return $this->handleAuthorizeNetGatewayData($gatewayData);
                }
            );

            Hooks::addFilter(
                'givewp_authorize_recurring_payment_description',
                DispatchGiveAuthorizeRecurringPaymentDescription::class,
                '__invoke',
                10,
                3
            );

            Hooks::addAction('give_authorize_webhook_payload', SubscriptionWebhooks::class, 'processWebhooks');

            Hooks::addAction(
                'givewp_authorize_event_handle_subscription_donations',
                HandleSubscriptionDonations::class,
                'run',
                10,
                2
            );

            Hooks::addAction(
                'givewp_authorize_event_subscription_active',
                SubscriptionActive::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction(
                'givewp_authorize_event_subscription_cancelled',
                SubscriptionCancelled::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction(
                'givewp_authorize_event_subscription_suspended',
                SubscriptionSuspended::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction(
                'givewp_authorize_event_subscription_completed',
                SubscriptionCompleted::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction(
                'givewp_authorize_event_subscription_expired',
                SubscriptionExpired::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction(
                'givewp_authorize_event_subscription_failing',
                SubscriptionFailing::class,
                '__invoke',
                10,
                2
            );
        }
    }

    /**
     * @since 2.2.0
     * @throws PaymentGatewayException
     */
    private function handleAuthorizeNetGatewayData($gatewayData)
    {
        if (is_array($gatewayData) && ! empty($gatewayData)) {
            $gatewayData['authorizeGatewayData'] = AuthorizeGatewayData::fromRequest($gatewayData);
        } else {
            $gatewayData['authorizeGatewayData'] = AuthorizeGatewayData::fromRequest(give_clean($_POST));
        }

        return $gatewayData;
    }

    /**
     * @since 2.5.0
     */
    private function registerPayPalStandardModule()
    {
        if (class_exists(PayPalStandard::class)) {
            add_filter(
                sprintf("givewp_gateway_%s_subscription_module", PayPalStandard::id()),
                static function () {
                    return PayPalStandardGatewaySubscriptionModule::class;
                }
            );

            add_filter(
                'give_subscription_can_cancel',
                static function (bool $ret, Give_Subscription $subscription) {
                    return give(PayPalStandardGatewayLegacyAdapter::class)->canCancelSubscription($ret, $subscription);
                }
                , 99, 2
            );

            // Add settings to PayPal standard
            add_filter('give_settings_gateways', static function ($settings) {
                return give(PayPalStandardGatewayLegacyAdapter::class)->settings($settings);
            });

            // In Give 2.9.0, PayPal Standard settings moved to "Payment Gateways -> PayPal -> PayPal standard"
            add_filter('give_get_settings_paypal_standard', static function ($settings) {
                return give(PayPalStandardGatewayLegacyAdapter::class)->settings($settings);
            });

            // v2 form validation
            add_action('give_checkout_error_checks', static function ($valid_data) {
                give(PayPalStandardGatewayLegacyAdapter::class)->checkoutErrors($valid_data);
            });

            // validate recurring times
            add_action('save_post_give_forms', static function ($form_id) {
                return give(PayPalStandardGatewayLegacyAdapter::class)->validatePaypalRecurringTimes($form_id);
            });

            // PayPal standard Webhooks
            give(PayPalStandardGatewayLegacyWebhooksAdapter::class)->processWebhooks();
        }
    }

    /**
     * @since 2.5.0
     */
    private function registerStripePaymentElementModule()
    {
        if (class_exists(StripePaymentElementGateway::class)) {
            add_filter(
                sprintf("givewp_gateway_%s_subscription_module", StripePaymentElementGateway::id()),
                static function () {
                    return StripePaymentElementGatewaySubscriptionModule::class;
                }
            );
        }
    }

    /**
     * @since 2.5.0
     */
    private function registerTestGatewayModule()
    {
        if (class_exists(TestGateway::class)) {
            add_filter(
                sprintf("givewp_gateway_%s_subscription_module", TestGateway::id()),
                static function () {
                    return TestGatewaySubscriptionModule::class;
                }
            );
        }

        if (class_exists(TestOffsiteGateway::class)) {
            add_filter(
                sprintf("givewp_gateway_%s_subscription_module", TestOffsiteGateway::id()),
                static function () {
                    return TestOffsiteGatewaySubscriptionModule::class;
                }
            );
        }
    }

    /**
     * @since 2.5.0
     */
    private function registerPayPalCommerceModule()
    {
        if (class_exists(PayPalCommerceGateway::class)) {
            Hooks::addFilter(
                'givewp_create_subscription_gateway_data_'  . PayPalCommerceGateway::id(),
                CreateSubscriptionGatewayData::class
            );
            Hooks::addFilter('give_recurring_modify_donation_data', ModifySubscriptionDonationData::class);

            // Register subscription module for PayPal Donations.
            add_filter(
                sprintf("givewp_gateway_%s_subscription_module", PayPalCommerceGateway::id()),
                static function () {
                    return PayPalCommerceSubscriptionModule::class;
                }
            );
        }
    }

    /**
     * @since 2.7.0
     */
    private function registerMollieModule()
    {
        if (class_exists(MollieGateway::class)) {
            $mollieGatewayId = MollieGateway::id();
            add_filter("givewp_gateway_{$mollieGatewayId}_subscription_module", function () {
                return MollieGatewaySubscriptionModule::class;
            });

            Hooks::addAction('givewp_mollie_webhook_request_handler', MollieSubscriptionWebhookRequestHandler::class);

            Hooks::addAction(
                'givewp_mollie_event_subscription_first_donation',
                SubscriptionFirstDonationCompleted::class
            );
            Hooks::addAction(
                'givewp_mollie_event_subscription_renewal_donation',
                SubscriptionRenewalDonationCreated::class,
                '__invoke',
                10,
                2
            );
        }
    }

    /**
     * @since 2.8.0
     *
     * @throws \Exception
     */
    private function registerBraintreeModule()
    {
        if (class_exists(BraintreeGateway::class)) {
            $braintreeGatewayId = BraintreeGateway::id();
            add_filter("givewp_gateway_{$braintreeGatewayId}_subscription_module", function () {
                return BraintreeGatewaySubscriptionModule::class;
            });

            add_filter("givewp_create_subscription_gateway_data_{$braintreeGatewayId}", function ($gatewayData) {
                return give(HandleBraintreeGatewayData::class)($gatewayData);
            });

            Hooks::addAction('admin_init', Notices::class, 'WebhookUrlWarning');

            Hooks::addAction('givewp_braintree_webhook_notification_handler',
                BraintreeSubscriptionWebhookNotificationHandler::class);

            Hooks::addAction('givewp_braintree_event_subscription_first_donation',
                SubscriptionFirstDonationCompleted::class);
            Hooks::addAction(
                'givewp_braintree_event_subscription_renewal_donation',
                SubscriptionRenewalDonationCreated::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction('givewp_braintree_event_subscription_active', SubscriptionActive::class);
            Hooks::addAction('givewp_braintree_event_subscription_cancelled', SubscriptionCancelled::class);
            Hooks::addAction('givewp_braintree_event_subscription_expired', SubscriptionExpired::class);
            Hooks::addAction('givewp_braintree_event_subscription_failing', SubscriptionFailing::class);
        }
    }

    /**
     * @since 2.9.0
     *
     * @throws \Exception
     */
    private function registerRazorpayModule()
    {
        if (class_exists(RazorpayGateway::class)) {
            $razorpayGatewayId = RazorpayGateway::id();
            add_filter("givewp_gateway_{$razorpayGatewayId}_subscription_module", function () {
                return RazorpayGatewaySubscriptionModule::class;
            });

            add_filter("givewp_create_subscription_gateway_data_{$razorpayGatewayId}", function ($gatewayData) {
                return give(HandleRazorpayGatewayData::class)($gatewayData);
            });

            Hooks::addAction('givewp_razorpay_webhook_notification',
                RazorpaySubscriptionWebhookNotificationHandler::class);

            Hooks::addAction('givewp_razorpay_event_subscription_cancelled', SubscriptionCancelled::class);
            Hooks::addAction(
                'givewp_razorpay_event_subscription_renewal_donation',
                SubscriptionRenewalDonationCreated::class,
                '__invoke',
                10,
                2
            );
        }
    }

    /**
     * @since 2.11.0
     *
     * @throws \Exception
     */
    private function registerBlinkPaymentModule(): void
    {
        if (class_exists(BlinkGateway::class)) {
            $blinkPaymentGatewayId = BlinkGateway::id();
            add_filter("givewp_gateway_{$blinkPaymentGatewayId}_subscription_module", static function () {
                return BlinkGatewaySubscriptionModule::class;
            });

            Hooks::addAction(
                'givewp_blink_event_subscription_renewal_donation',
                SubscriptionRenewalDonationCreated::class,
                '__invoke',
                10,
                2
            );
            Hooks::addAction('givewp_blink_event_subscription_failing', SubscriptionFailing::class);
        }
    }
}
