<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Braintree\Webhooks;

use Braintree\WebhookNotification;
use Give\Framework\Support\Facades\ActionScheduler\AsBackgroundJobs;

/**
 * @since 2.8.0
 */
class BraintreeSubscriptionWebhookNotificationHandler
{
    /**
     * @since 2.8.0
     */
    public function __invoke(WebhookNotification $braintreeWebhookNotification)
    {
        /**
         * @see https://developer.paypal.com/braintree/docs/reference/general/webhooks/subscription/php
         */
        switch ($braintreeWebhookNotification->kind) {
            case WebhookNotification::SUBSCRIPTION_CHARGED_SUCCESSFULLY:
                $totalTransactions = count($braintreeWebhookNotification->subscription->transactions);
                if ($totalTransactions === 1) {
                    $this->setSubscriptionFirstPaymentAsyncEvent($braintreeWebhookNotification->subscription->transactions[0]->id);
                } elseif ($totalTransactions > 1) {
                    $this->setSubscriptionRenewalPaymentAsyncEvent($braintreeWebhookNotification->subscription->id,
                        $braintreeWebhookNotification->subscription->transactions[0]->id);
                }
                break;
            case WebhookNotification::SUBSCRIPTION_WENT_ACTIVE:
                $this->setSubscriptionActiveAsyncEvent($braintreeWebhookNotification->subscription->id);
                break;
            case WebhookNotification::SUBSCRIPTION_CANCELED:
                $this->setSubscriptionCancelledAsyncEvent($braintreeWebhookNotification->subscription->id);
                break;
            case WebhookNotification::SUBSCRIPTION_EXPIRED:
                $this->setSubscriptionExpiredAsyncEvent($braintreeWebhookNotification->subscription->id);
                break;
            case WebhookNotification::SUBSCRIPTION_BILLING_SKIPPED:
            case WebhookNotification::SUBSCRIPTION_CHARGED_UNSUCCESSFULLY:
            case WebhookNotification::SUBSCRIPTION_WENT_PAST_DUE:
                $this->setSubscriptionFailingAsyncEvent($braintreeWebhookNotification->subscription->id);
                break;
        }
    }

    /**
     * @since 2.8.0
     */
    public function setSubscriptionFirstPaymentAsyncEvent(string $gatewayTransactionId)
    {
        $this->enqueueAsyncAction('givewp_braintree_event_subscription_first_donation',
            [$gatewayTransactionId]);
    }

    /**
     * @since 2.8.0
     */
    public function setSubscriptionRenewalPaymentAsyncEvent(
        string $gatewaySubscriptionId,
        string $gatewayTransactionId
    ) {
        $this->enqueueAsyncAction('givewp_braintree_event_subscription_renewal_donation',
            [$gatewaySubscriptionId, $gatewayTransactionId]);
    }

    /**
     * @since 2.8.0
     */
    private function setSubscriptionActiveAsyncEvent(string $gatewaySubscriptionId)
    {
        $this->enqueueAsyncAction('givewp_braintree_event_subscription_active', [$gatewaySubscriptionId]);
    }

    /**
     * @since 2.8.0
     */
    private function setSubscriptionCancelledAsyncEvent(string $gatewaySubscriptionId)
    {
        $this->enqueueAsyncAction('givewp_braintree_event_subscription_cancelled', [$gatewaySubscriptionId]);
    }

    /**
     * @since 2.8.0
     */
    private function setSubscriptionExpiredAsyncEvent(string $gatewaySubscriptionId)
    {
        $this->enqueueAsyncAction('givewp_braintree_event_subscription_expired', [$gatewaySubscriptionId]);
    }

    /**
     * @since 2.8.0
     */
    private function setSubscriptionFailingAsyncEvent(string $gatewaySubscriptionId)
    {
        $this->enqueueAsyncAction('givewp_braintree_event_subscription_failing', [$gatewaySubscriptionId]);
    }

    /**
     * @since 2.8.0
     */
    private function enqueueAsyncAction(string $hook, array $args)
    {
        AsBackgroundJobs::enqueueAsyncAction($hook, $args, 'give-recurring-braintree');
    }
}
