<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions;

use Exception;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Subscriptions\Models\Subscription;
use GiveBlink\API\BlinkApi;
use GiveBlink\DataTransferObjects\GatewayDataDTO;
use GiveBlink\DataTransferObjects\PaymentIntentDTO;
use GiveBlink\Helpers\GatewayHelper;

/**
 * @since 2.11.0
 */
class CreateBlinkSubscription
{
    /**
     * @since 2.11.0
     *
     * @throws PaymentGatewayException;
     */
    public function __invoke(Subscription $subscription, GatewayDataDTO $gatewayData, PaymentIntentDTO $intent): array
    {
        try {
            $paymentData = $this->buildSubscriptionData($subscription, $gatewayData, $intent);

            return BlinkApi::client()->processRepeatPayment($paymentData);
        } catch (Exception $e) {
            throw new PaymentGatewayException(
                sprintf(
                    __('[Blink] Failed to create subscription. API Error: %s', 'give-blink'),
                    $e->getCode() . ' - ' . $e->getMessage()
                )
            );
        }
    }

    /**
     * Build the payment data array for processing.
     *
     * @since 2.11.0
     */
    private function buildSubscriptionData(
        Subscription $subscription,
        GatewayDataDTO $gatewayData,
        PaymentIntentDTO $intent
    ): array {
        $isLimitedInstallments = $subscription->installments > 0;

        $data = [
            'payment_type' => 'fixed_schedule',
            'payment_intent' => $intent->paymentIntent,
            'paymentToken' => $gatewayData->paymentMethodData['paymentDetails']['paymentToken'],
            'type' => '1',
            'customer_email' => $subscription->initialDonation()->email,
            'customer_name' => $gatewayData->paymentMethodData['nameOnCard'],
            'reference' => 'GIVE-SUB-' . $subscription->id,
            'currency' => $subscription->initialDonation()->amount->getCurrency()->getCode(),
            'merchant_data' => json_encode([
                'donation_id' => $subscription->initialDonation()->id,
                'subscription_id' => $subscription->id,
                'payment_method' => $gatewayData->paymentMethod,
            ]),
            'frequency' => $subscription->period . 's',
            'frequency_duration' => $subscription->frequency,
            'recurring_amount' => (float)$subscription->initialDonation()->amount->formatToDecimal(),
            'is_limited_installments' => $isLimitedInstallments,
            'device_timezone' => $gatewayData->deviceDetails['device_timezone'],
            'device_capabilities' => $gatewayData->deviceDetails['device_capabilities'],
            'device_accept_language' => $gatewayData->deviceDetails['device_accept_language'],
            'device_screen_resolution' => $gatewayData->deviceDetails['device_screen_resolution'],
            'remote_address' => GatewayHelper::getClientIp(),
        ];

        if ($isLimitedInstallments) {
            $data['installments'] = $subscription->installments;
        }

        return $data;
    }

    /**
     * Possible values: ... months ... weeks ... days
     * Example: 1 month or 14 days
     * IMPORTANT: The maximum interval is 1 year (12 months, 52 weeks or 365 days).
     *
     * @since 2.11.0
     */
    private function getSubscriptionInterval(Subscription $subscription): string
    {
        $multiple = $subscription->frequency > 1;
        $interval = '1 month';

        if ($subscription->period->isDay()) {
            $interval = $subscription->frequency . ($multiple ? ' days' : ' day');
        } elseif ($subscription->period->isWeek()) {
            $interval = $subscription->frequency . ($multiple ? ' weeks' : ' week');
        } elseif ($subscription->period->isMonth()) {
            $interval = $subscription->frequency . ($multiple ? ' months' : ' month');
        } elseif ($subscription->period->isQuarter()) {
            $interval = $subscription->frequency * 3 . ' months';
        } elseif ($subscription->period->isYear()) {
            $interval = '12 months';
        }

        return $interval;
    }
}
