<?php

namespace GiveAuthorizeNet\Webhooks;

use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use GiveAuthorizeNet\DataTransferObjects\ApiAccessData;
use GiveAuthorizeNet\ValueObjects\WebhooksServerType;
use stdClass;
use WP;

/**
 * @since 2.0.0 Webhooks refactoring to use this dedicated class, better logs, models, and action scheduler
 */
class WebhooksProcessors
{
    /**
     * AJAX check webhooks.
     *
     * @since 2.0.0 Use ApiAccessData class
     */
    public function ajaxCheckAuthorizeWebhooks()
    {
        $apiAccessData = ApiAccessData::fromOptions();

        $data = [
            'live_webhooks_setup' => false,
            'sandbox_webhooks_setup' => false,
        ];

        if ($this->isLiveWebhooksUnconfigured($apiAccessData)) {
            $data['live_webhooks_setup'] = 'unconfigured';
        } elseif ($this->isLiveWebhooksSetup($apiAccessData)) {
            $data['live_webhooks_setup'] = true;
        } elseif ($this->isLiveWebhooksNotSetup($apiAccessData)) {
            $data['live_webhooks_setup'] = (new SetupWebhooks())($apiAccessData->liveLoginId,
                $apiAccessData->liveTransactionKey, WebhooksServerType::PRODUCTION);
        }

        if ($this->isSandboxWebhooksUnconfigured($apiAccessData)) {
            $data['sandbox_webhooks_setup'] = 'unconfigured';
        } elseif ($this->isSandboxWebhooksSetup($apiAccessData)) {
            $data['sandbox_webhooks_setup'] = true;
        } elseif ($this->isSandboxWebhooksNotSetup($apiAccessData)) {
            $data['sandbox_webhooks_setup'] = (new SetupWebhooks())($apiAccessData->sandboxLoginId,
                $apiAccessData->sandboxTransactionKey, WebhooksServerType::SANDBOX);
        }

        wp_send_json_success($data);

        wp_die();
    }

    /**
     * Force recheck of webhooks.
     *
     * @since 1.3
     */
    public function ajaxForceRecheckWebhooks()
    {
        give_delete_option('give_authorize_live_webhooks_setup');
        give_delete_option('give_authorize_sandbox_webhooks_setup');

        $this->ajaxCheckAuthorizeWebhooks();
    }

    /**
     * Listen for Authorize.net webhook events.
     *
     * @access      public
     * @since       1.3
     *
     * @param WP $query
     */
    public function eventListener($query)
    {
        // Must be a GiveWP Authorize.net listener to proceed.
        if ('give-authorize-webhook-listener' !== $query->request) {
            return $query;
        }

        // Retrieve the request's body and parse it as JSON.
        $body = @file_get_contents('php://input');
        // Decode JSON into object.
        $eventJson = json_decode($body);

        // Process the webhooks.
        $this->processWebhooks($eventJson);
    }

    /**
     * Process Authorize.Net Webhooks.
     *
     * @see        https://developer.authorize.net/api/reference/features/webhooks.html#Event_Types_and_Payloads
     *
     * @since 2.0.0 Use PaymentGatewayLog class and action scheduler and handle "net.authorize.payment.authcapture.created" event
     * @since      1.3
     *
     * @param stdClass $eventJson
     */
    public function processWebhooks($eventJson)
    {
        if ( ! isset($eventJson->eventType) || ! isset($eventJson->payload->id)) {
            status_header(500);

            PaymentGatewayLog::error(
                '[Authorize.net] An error occurred while processing a webhook.',
                [
                    'Event Json' => $eventJson,
                ]
            );

            die('-1'); // Failed
        }

        status_header(200);

        PaymentGatewayLog::debug(
            sprintf('[Authorize.Net] Webhooks: Trigger %s event for %s id %s.',
                $eventJson->eventType,
                $eventJson->payload->entityName,
                $eventJson->payload->id),
            [
                '$eventJson' => $eventJson,
            ]
        );

        /**
         * Fire the action
         *
         * @since 2.0.0
         *
         * @param stdClass $eventJson
         */
        do_action("give_authorize_webhook_payload", $eventJson);

        switch (strtolower($eventJson->eventType)) {
            // Refunds and voids both receive refunds in Give.
            case 'net.authorize.payment.void.created':
                as_enqueue_async_action('givewp_authorize_event_donation_refunded',
                    [$eventJson->payload->id, __('[Authorize.Net] Transaction voided.', 'give-authorize')],
                    'give-authorize');
                break;
            case 'net.authorize.payment.refund.created':
                as_enqueue_async_action('givewp_authorize_event_donation_refunded',
                    [$eventJson->payload->id, __('[Authorize.Net] Transaction refunded', 'give-authorize')],
                    'give-authorize');
                break;
            case 'net.authorize.payment.fraud.approved':
                as_enqueue_async_action('givewp_authorize_event_donation_completed',
                    [
                        $eventJson->payload->id,
                        __('[Authorize.Net] Transaction approved by the fraud filter.', 'give-authorize'),
                    ],
                    'give-authorize');
                break;
            case 'net.authorize.payment.fraud.declined':
                as_enqueue_async_action('givewp_authorize_event_donation_failed',
                    [
                        $eventJson->payload->id,
                        __('[Authorize.Net] Transaction declined by the fraud filter.', 'give-authorize'),
                    ],
                    'give-authorize');
                break;
            case 'net.authorize.payment.authcapture.created':
                as_enqueue_async_action('givewp_authorize_event_donation_completed',
                    [$eventJson->payload->id, __('[Authorize.Net] Transaction approved.', 'give-authorize')],
                    'give-authorize');
                break;
            default:
                break;
        }

        /**
         * Fire the action
         *
         * @param stdClass $eventJson
         */
        do_action("give_authorize_event_{$eventJson->eventType}", $eventJson);

        die('1'); // Completed successfully
    }

    /**
     * @since 2.0.0
     */
    private function isLiveWebhooksUnconfigured(ApiAccessData $apiAccessData): bool
    {
        return empty($apiAccessData->liveLoginId) || empty($apiAccessData->liveTransactionKey);
    }

    /**
     * @since 2.0.0
     */
    private function isLiveWebhooksSetup(ApiAccessData $apiAccessData): bool
    {
        return give_is_test_mode() && $apiAccessData->liveWebhooksSetup;
    }

    /**
     * @since 2.0.0
     */
    private function isLiveWebhooksNotSetup(ApiAccessData $apiAccessData): bool
    {
        return ! empty($apiAccessData->liveLoginId) && ! empty($apiAccessData->liveTransactionKey);
    }

    /**
     * @since 2.0.0
     */
    private function isSandboxWebhooksUnconfigured(ApiAccessData $apiAccessData): bool
    {
        return empty($apiAccessData->sandboxLoginId) || empty($apiAccessData->sandboxTransactionKey);
    }

    /**
     * @since 2.0.0
     */
    private function isSandboxWebhooksSetup(ApiAccessData $apiAccessData): bool
    {
        return give_is_test_mode() && $apiAccessData->sandboxWebhooksSetup;
    }

    /**
     * @since 2.0.0
     */
    private function isSandboxWebhooksNotSetup(ApiAccessData $apiAccessData): bool
    {
        return ! empty($apiAccessData->sandboxLoginId) && ! empty($apiAccessData->sandboxTransactionKey);
    }
}
