<?php
namespace GiveAuthorizeNet\Webhooks;

use Exception;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use GiveAuthorizeNet\ValueObjects\WebhooksServerType;
use JohnConde\Authnet\AuthnetApiFactory;

/**
 * @since 2.0.0
 */
class SetupWebhooks
{
    /**
     * Setup webhooks with Authorize.net and website.
     *
     * @see        http://www.johnconde.net/blog/handling-authorize-net-webhooks-with-php/
     *
     * @since 2.0.0 Refactored to use this dedicated class, better logs, and WebhooksServerType for the server param
     * @since      1.3
     *
     * @return bool|string
     */
    public function __invoke(string $loginId, string $transactionKey, int $server)
    {
        $listenerUrl = home_url('/give-authorize-webhook-listener/');

        try {
            $request = AuthnetApiFactory::getWebhooksHandler($loginId, $transactionKey, $server);

            // Get AuthnetWebhooksResponse object.
            $webhooksResponse = $request->getWebhooks();

            // Get a list of webhooks from the AuthnetWebhooksResponse object.
            $webhooksCheck = $webhooksResponse->getWebhooks();

            // Check if webhooks have already been configured for this URL.
            foreach ($webhooksCheck as $webhook) {
                // Already configured. Bounce out.
                if ($webhook->getUrl() === $listenerUrl) {
                    // Successful webhook setup.
                    if (WebhooksServerType::PRODUCTION === $server) {
                        give_update_option('give_authorize_live_webhooks_setup', true);
                        give_update_option('give_authorize_live_webhooks_id', $webhook->getWebhooksId());
                        give_delete_option('give_authorize_live_webhooks_no_signature_key');
                    } elseif (WebhooksServerType::SANDBOX === $server) {
                        give_update_option('give_authorize_sandbox_webhooks_setup', true);
                        give_update_option('give_authorize_sandbox_webhooks_id', $webhook->getWebhooksId());
                        give_delete_option('give_authorize_sandbox_webhooks_no_signature_key');
                    }

                    return true;
                }
            }

            // Ok, now create the webhooks for this URL.
            $response = (new CreateWebhooks())($listenerUrl, $loginId, $transactionKey, $server);

            if ('active' === $response->getStatus()) {
                // Successful webhook setup.
                if (WebhooksServerType::PRODUCTION) {
                    give_update_option('give_authorize_live_webhooks_setup', true);
                    give_update_option('give_authorize_live_webhooks_id', $response->getWebhooksId());
                    give_delete_option('give_authorize_live_webhooks_no_signature_key');
                } elseif (WebhooksServerType::SANDBOX) {
                    give_update_option('give_authorize_sandbox_webhooks_setup', true);
                    give_update_option('give_authorize_sandbox_webhooks_id', $response->getWebhooksId());
                    give_delete_option('give_authorize_sandbox_webhooks_no_signature_key');
                }
            }

            return true;
        } catch (Exception $e) {
            PaymentGatewayLog::error(
                '[Authorize.net] Attempt to set up webhooks failed.',
                [
                    'Code' => $e->getCode(),
                    'Message' => $e->getMessage(),
                ]
            );

            // Common error is not having the signature key setup. Flag notice and only show for live sites.
            if (strpos($e->getMessage(), 'Please generate a signature key') !== false) {
                // Live and sandbox.
                if (WebhooksServerType::PRODUCTION) {
                    give_update_option('give_authorize_live_webhooks_no_signature_key', true);
                } else {
                    give_update_option('give_authorize_sandbox_webhooks_no_signature_key', true);
                }
            }

            return 'error';
        } // End try().
    }
}
