<?php

namespace GiveAuthorizeNet;

use Give\Framework\Migrations\MigrationsRegister;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\PaymentGatewayRegister;
use Give\Helpers\Hooks;
use GiveAuthorizeNet\Admin\Notices;
use GiveAuthorizeNet\DataTransferObjects\AuthorizeGatewayData;
use GiveAuthorizeNet\EventHandlers\DonationCompleted;
use GiveAuthorizeNet\EventHandlers\DonationFailed;
use GiveAuthorizeNet\EventHandlers\DonationRefunded;
use GiveAuthorizeNet\Gateway\AcceptJs;
use GiveAuthorizeNet\Gateway\CreditCardGateway;
use GiveAuthorizeNet\Gateway\eCheckGateway;
use GiveAuthorizeNet\LegacyListeners\DispatchGiveAuthorizeOneTimePaymentDescription;
use GiveAuthorizeNet\Migrations\UpdateLivePublicClientKey;
use GiveAuthorizeNet\Migrations\UpdateSandboxPublicClientKey;
use GiveAuthorizeNet\Webhooks\WebhooksProcessors;

/**
 * @since 1.5.0
 */
class ServiceProvider implements \Give\ServiceProviders\ServiceProvider
{

    /**
     * @inerhitDoc
     */
    public function register()
    {
    }

    /**
     * @since 2.0.3 Use givewp_authorize_one_time_payment_description filter
     * @since      2.0.0 Refactoring to use the gateway API, new Webhook class, better logs and actions scheduler
     *
     * @inerhitDoc
     * @throws PaymentGatewayException
     */
    public function boot()
    {
        $this->registerMigrations();
        $this->registerWebhooksProcessors();
        $this->registerActionSchedulerEventHandlers();
        $this->registerRefundCheckbox();
        $this->registerAdminNotices();

        Hooks::addFilter('givewp_authorize_one_time_payment_description',
            DispatchGiveAuthorizeOneTimePaymentDescription::class, '__invoke', 10, 2);

        add_action('givewp_register_payment_gateway', function (PaymentGatewayRegister $registrar) {
            $registrar->registerGateway(CreditCardGateway::class);
            $registrar->registerGateway(eCheckGateway::class);
        });
        $creditCardId = CreditCardGateway::id();
        add_filter("givewp_create_payment_gateway_data_{$creditCardId}", function ($gatewayData) {
            return $this->handleAuthorizeNetGatewayData($gatewayData);
        });
        $eCheckId = eCheckGateway::id();
        add_filter("givewp_create_payment_gateway_data_{$eCheckId}", function ($gatewayData) {
            return $this->handleAuthorizeNetGatewayData($gatewayData);
        });
        Hooks::addAction('give_after_cc_fields', AcceptJs::class, 'addHiddenFields');
    }

    /**
     * @since 2.0.0
     */
    private function registerMigrations()
    {
        /* @var MigrationsRegister $migrationRegister */
        $migrationRegister = give(MigrationsRegister::class);

        if (give_is_test_mode()) {
            $migrationRegister->addMigration(UpdateSandboxPublicClientKey::class);
        } else {
            $migrationRegister->addMigration(UpdateLivePublicClientKey::class);
        }
    }

    /**
     * @since 2.0.0
     */
    private function registerWebhooksProcessors()
    {
        Hooks::addAction('wp_ajax_check_authorize_webhooks', WebhooksProcessors::class, 'ajaxCheckAuthorizeWebhooks');
        Hooks::addAction('wp_ajax_hard_check_authorize_webhooks', WebhooksProcessors::class,
            'ajaxForceRecheckWebhooks');
        Hooks::addAction('parse_request', WebhooksProcessors::class, 'eventListener');
    }

    /**
     * @since 2.0.0
     */
    private function registerActionSchedulerEventHandlers()
    {
        Hooks::addAction('givewp_authorize_event_donation_refunded', DonationRefunded::class, 'setStatus', 10,
            2);
        Hooks::addAction('givewp_authorize_event_donation_completed', DonationCompleted::class, 'setStatus', 10,
            2);
        Hooks::addAction('givewp_authorize_event_donation_failed', DonationFailed::class, 'setStatus', 10,
            2);
    }

    /**
     * @since 2.0.0
     */
    private function registerRefundCheckbox()
    {
        Hooks::addAction('give_view_donation_details_totals_after', CreditCardGateway::class, 'addOptRefundCheckbox',
            11);
        Hooks::addAction('give_update_payment_status', CreditCardGateway::class, 'refundDonationAdapter', 10, 3);

        Hooks::addAction('give_view_donation_details_totals_after', eCheckGateway::class, 'addOptRefundCheckbox', 11);
        Hooks::addAction('give_update_payment_status', eCheckGateway::class, 'refundDonationAdapter', 10, 3);
    }

    /**
     * @since 2.0.0
     */
    private function registerAdminNotices()
    {
        Hooks::addAction('admin_init', Notices::class, 'MissingClientPublicKey');
    }

    /**
     * @since 2.0.0
     * @throws PaymentGatewayException
     */
    private function handleAuthorizeNetGatewayData($gatewayData)
    {
        if ( ! empty($gatewayData)) { //V3 Forms
            $gatewayData['authorizeGatewayData'] = AuthorizeGatewayData::fromRequest($gatewayData);
        } else {
            $gatewayData['authorizeGatewayData'] = AuthorizeGatewayData::fromRequest(give_clean($_POST));
        }

        return $gatewayData;
    }
}
