import {CSSProperties, useEffect, useState} from 'react';
import {__} from '@wordpress/i18n';
import {Gateway} from './Interfaces';

type AuthorizeGatewaySettings = {
    acceptJsUrl: string;
    clientPublicKey: string;
    apiLoginId: string;
    accountTypes: Array<{[key: string]: string}>;
    checkSampleImageUrl: string;
};

type AuthorizeBankInfo = {
    routingNumber: string;
    accountNumber: string;
    nameOnAccount: string;
    accountType: string;
};

let settings: AuthorizeGatewaySettings;

let bankData: AuthorizeBankInfo = {
    routingNumber: '',
    accountNumber: '',
    nameOnAccount: '',
    accountType: '',
};

let acceptJsDataDescriptor = '';
let acceptJsDataValue = '';

const eCheckGateway: Gateway = {
    id: 'authorize_echeck',
    initialize() {
        settings = this.settings;
    },
    beforeCreatePayment: async function (values: FormData): Promise<object> {
        const authData = {
            apiLoginID: settings.apiLoginId,
            clientKey: settings.clientPublicKey,
        };

        const secureData = {
            authData: authData,
            bankData: bankData,
        };

        await new Promise(function (resolve) {
            // @ts-ignore
            Accept.dispatchData(secureData, function (response) {
                let errorMessages = '';
                try {
                    if (response.messages.resultCode === 'Error') {
                        let i = 0;
                        while (i < response.messages.message.length) {
                            errorMessages += `${response.messages.message[i].code} : ${response.messages.message[i].text} \r\n`;
                            i = i + 1;
                        }
                    } else {
                        acceptJsDataDescriptor = response.opaqueData.dataDescriptor
                            ? response.opaqueData.dataDescriptor
                            : '';
                        acceptJsDataValue = response.opaqueData.dataValue ? response.opaqueData.dataValue : '';
                    }
                } catch (error) {
                    throw new Error(error);
                }

                if (errorMessages != '') {
                    throw new Error(errorMessages);
                }

                resolve('Accept.dispatchData');
            });
        });

        return {
            give_authorize_data_descriptor: acceptJsDataDescriptor,
            give_authorize_data_value: acceptJsDataValue,
        };
    },
    Fields() {
        // @ts-ignore
        const {useWatch} = window.givewp.form.hooks;
        const gatewayId = useWatch({name: 'gatewayId'});
        const isCurrentGatewayActive = gatewayId === eCheckGateway.id;

        if (!isCurrentGatewayActive) {
            return;
        }

        const script = document.createElement('script');
        script.src = settings.acceptJsUrl;

        document.body.appendChild(script);

        return <BankInfoFields />;
    },
};

const BankInfoFields = () => {
    const [bankInfoData, setBankInfoData] = useState<AuthorizeBankInfo>({
        routingNumber: '',
        accountNumber: '',
        nameOnAccount: '',
        accountType: Object.keys(settings.accountTypes)[0],
    });

    useEffect(() => {
        bankData = bankInfoData;
    }, [bankInfoData]);

    const styles = {
        img: {
            display: 'block',
            margin: '0 auto',
        },
        fieldset: {
            display: 'grid',
            gridTemplateColumns: 'repeat(auto-fit, minmax(200px, auto))',
            gap: '1rem',
            marginTop: '1.5rem',
        },
        legend: {
            marginBottom: '1.5rem',
        },
        label: {
            position: 'initial',
        } as CSSProperties,
        input: {
            borderColor: 'initial',
            marginTop: '0.5rem',
        },
        select: {
            lineHeight: 'initial',
            padding: '0.8rem',
            marginTop: '0.4rem',
        },
    };

    const requiredAsterisk = (
        <span className="givewp-field-required" title={__('Field Required', 'give-authorize')}>
            *
        </span>
    );

    return (
        <div>
            <img style={styles.img} src={settings.checkSampleImageUrl} alt={__('Check Sample', 'give-authorize')} />
            <fieldset style={styles.fieldset}>
                <legend style={styles.legend}>{__('Bank Details', 'give-authorize')}</legend>

                <label style={styles.label}>
                    {__('Routing Number', 'give-authorize')} {requiredAsterisk}
                    <input
                        style={styles.input}
                        type="tel"
                        autoComplete="off"
                        name="routing_number"
                        id="routing_number"
                        placeholder={__('Routing Number', 'give-authorize')}
                        required={true}
                        aria-required="true"
                        tabIndex={1}
                        onChange={(event) => setBankInfoData({...bankInfoData, routingNumber: event.target.value})}
                    />
                </label>

                <label style={styles.label}>
                    {__('Account Number', 'give-authorize')} {requiredAsterisk}
                    <input
                        style={styles.input}
                        type="tel"
                        autoComplete="off"
                        name="account_number"
                        id="account_number"
                        placeholder={__('Account Number', 'give-authorize')}
                        required={true}
                        aria-required="true"
                        tabIndex={1}
                        onChange={(event) => setBankInfoData({...bankInfoData, accountNumber: event.target.value})}
                    />
                </label>

                <label style={styles.label}>
                    {__('Name on Account', 'give-authorize')} {requiredAsterisk}
                    <input
                        style={styles.input}
                        type="text"
                        autoComplete="off"
                        name="name_on_account"
                        id="name_on_account"
                        placeholder={__('Name on Account', 'give-authorize')}
                        required={true}
                        aria-required="true"
                        tabIndex={1}
                        onChange={(event) => setBankInfoData({...bankInfoData, nameOnAccount: event.target.value})}
                    />
                </label>

                <label style={styles.label}>
                    {__('Account Type', 'give-authorize')} {requiredAsterisk}
                    <select
                        style={styles.select}
                        required={true}
                        name="account_type"
                        id="account_type"
                        tabIndex={1}
                        onChange={(event) => setBankInfoData({...bankInfoData, accountType: event.target.value})}
                    >
                        {Object.keys(settings.accountTypes).map((key) => (
                            <option key={key} value={key}>
                                {settings.accountTypes[key]}
                            </option>
                        ))}
                    </select>
                </label>
            </fieldset>
        </div>
    );
};

// @ts-ignore
window.givewp.gateways.register(eCheckGateway);
