<?php

namespace GiveAuthorizeNet\Gateway;

use Give\Framework\PaymentGateways\PaymentGateway;
use GiveAuthorizeNet\Exceptions\InvalidCredentialsException;
use GiveAuthorizeNet\Gateway\Traits\CreatePaymentGateway;
use GiveAuthorizeNet\Gateway\Traits\HasScriptAssetFile;
use GiveAuthorizeNet\Gateway\Traits\RefundDonationGateway;

/**
 * Unlike payment card transactions, eCheck transactions do not receive immediate authorization from the
 * customer bank. Like paper checks, eCheck transactions must clear the customer's bank before funding,
 * and merchants might wait up to seven days after submitting eCheck transactions to receive the funds.
 *
 * @see https://developer.authorize.net/api/reference/features/echeck.html
 *
 * @since 2.0.0
 */
class eCheckGateway extends PaymentGateway
{
    use CreatePaymentGateway;
    use RefundDonationGateway;
    use HasScriptAssetFile;

    /**
     * @since 2.0.0
     *
     * @throws InvalidCredentialsException
     */
    public function getLegacyFormFieldMarkup(int $formId, array $args): string
    {
        ob_start();

        AcceptJs::loadScript($formId, false);

        if ( ! give_is_setting_enabled(give_get_option('authorize_collect_billing'))) {
            remove_action('give_after_cc_fields', 'give_default_cc_address_fields');
        }

        $this->bankBillingFields($formId);

        $form = ob_get_clean();

        return $form;
    }

    /**
     * @since 2.0.0
     */
    public static function id(): string
    {
        return 'authorize_echeck';
    }

    /**
     * @since 2.0.0
     */
    public function getId(): string
    {
        return self::id();
    }

    /**
     * @since 2.0.0
     */
    public function getName(): string
    {
        return __('Authorize.Net - eCheck', 'give-authorize');
    }

    /**
     * @since 2.0.0
     */
    public function getPaymentMethodLabel(): string
    {
        return __('Authorize.Net - eCheck', 'give-authorize');
    }

    /**
     * Adds form fields necessary for eCheck payment
     *
     * @since 2.0.0
     */
    public function bankBillingFields(int $formId)
    {
        if ( ! give_is_setting_enabled(give_get_option('authorize_collect_billing'))) {
            remove_action('give_after_cc_fields', 'give_default_cc_address_fields');
        }

        ?>

        <div id="gateway-authorize-echeck-wrap" class="gateway-authorize-echeck" style="position: relative; zoom: 1;">

            <fieldset id="give_checkout_bank_details">
                <legend>
                    <?php
                    /**
                     * Filter the legend title.
                     */
                    echo apply_filters('give_checkout_bank_details', __('Bank Details', 'give-authorize'));
                    ?>
                </legend>

                <div class="give-check-sample">
                    <img src="<?php
                    echo GIVE_AUTHORIZE_PLUGIN_URL . '/assets/images/check-sample.jpg'; ?>" alt="<?php
                    echo __('Check Sample', 'give-authorize') ?>">
                </div>

                <div class="ga-column-wrap">

                    <!-- Routing Number input field -->
                    <p>
                        <label>
                            <?php
                            esc_html_e('Routing Number', 'give-authorize'); ?>
                            <?php
                            echo Give()->tooltips->render_help(__('The bank\'s routing number.', 'give-authorize')); ?>
                        </label>
                        <input required type="text" name="routing-number" id="routing-number-<?php
                        echo $formId ?>"
                               placeholder="<?php
                               echo esc_attr('Routing Number*', 'give-authorize'); ?>">
                    </p>

                    <!-- Account Number input field -->
                    <p>
                        <label>
                            <?php
                            esc_html_e('Account Number', 'give-authorize'); ?>
                            <?php
                            echo Give()->tooltips->render_help(__('The bank account number.', 'give-authorize')); ?>
                        </label>
                        <input required type="text" name="account-number" id="account-number-<?php
                        echo $formId ?>"
                               placeholder="<?php
                               echo esc_attr('Account Number*', 'give-authorize'); ?>">
                    </p>

                </div>

                <div class="ga-column-wrap">
                    <!-- Name on the Account input field -->
                    <p>
                        <label>
                            <?php
                            esc_html_e('Name on Account', 'give-authorize'); ?>
                            <?php
                            echo Give()->tooltips->render_help(__('The name of the person who holds the bank account.',
                                'give-authorize')); ?>
                        </label>
                        <input required type="text" name="name-on-account" id="name-on-account-<?php
                        echo $formId ?>"
                               placeholder="<?php
                               echo esc_attr('Name on Account*', 'give-authorize'); ?>">
                    </p>

                    <!-- Account Types select fields -->
                    <p>
                        <label>
                            <?php
                            esc_html_e('Account Type', 'give-authorize'); ?>
                            <?php
                            echo Give()->tooltips->render_help(__('The type of bank account.', 'give-authorize')); ?>
                        </label>
                        <?php
                        $this->selectField('account-type', 'account-type-' . $formId); ?>
                    </p>
                </div>
            </fieldset>

        </div>

        <?php

        do_action('give_after_cc_fields', $formId);
    }

    /**
     * List of account types
     *
     * @since 2.0.0
     */
    public static function getAccountTypes()
    {
        /**
         * Filter to add another account type.
         */
        return apply_filters('give_add_echeck_account_type', [
            'checking' => __('Checking', 'give-authorize'),
            'savings' => __('Savings', 'give-authorize'),
            'businessChecking' => __('Business Checking', 'give-authorize'),
        ]);
    }

    /**
     * Generates select dropdown list for bank details.
     *
     * @since 2.0.0
     */
    public function selectField(string $name, string $id)
    {
        $accountTypes = self::getAccountTypes();

        if ( ! empty($accountTypes)) {
            printf('<select required name="%s" id="%s">', esc_attr($name), esc_attr($id));

            foreach ($accountTypes as $attr => $value) {
                printf('<option value="%1$s">%2$s</option>', esc_attr($attr), esc_html($value));
            }

            printf('</select>');
        }
    }

    /**
     * @since 3.0
     * @throws InvalidCredentialsException
     */
    public function formSettings(int $formId): array
    {
        return array_merge(
            AcceptJs::getApiAccessData(),
            [
                'accountTypes' => self::getAccountTypes(),
                'checkSampleImageUrl' => trailingslashit(GIVE_AUTHORIZE_PLUGIN_URL) . 'assets/images/check-sample.jpg',
            ]
        );
    }

    /**
     * @since 3.0
     */
    public function enqueueScript(int $formId)
    {
        $assets = $this->getScriptAsset( trailingslashit( GIVE_AUTHORIZE_PLUGIN_DIR ) . 'build/eCheckGateway.asset.php' );

        wp_enqueue_script(
            self::id(),
            trailingslashit( GIVE_AUTHORIZE_PLUGIN_URL ) . 'build/eCheckGateway.js',
            $assets['dependencies'],
            $assets['version'],
            true
        );
    }
}
