<?php

namespace GiveAuthorizeNet\Gateway\Traits;

use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\Exceptions\Primitives\Exception;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use GiveAuthorizeNet\Actions\RefundTransaction;
use GiveAuthorizeNet\Exceptions\InvalidCredentialsException;

/**
 * @since 2.0.0
 */
trait RefundDonationGateway
{
    /**
     * @since 2.0.0
     *
     * @throws Exception
     */
    public function refundDonation(Donation $donation)
    {
        try {
            $this->refundTransaction($donation);
            $donation->status = DonationStatus::REFUNDED();
            $donation->save();

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(
                    __('Donation refunded in Authorize.Net for transaction ID: %s', 'give-authorize'),
                    $donation->gatewayTransactionId
                ),
            ]);
        } catch (PaymentGatewayException $exception) {
            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(
                    __(
                        'Error! Donation %s was NOT refunded. Find more details on the error in the logs at Donations > Tools > Logs. To refund the donation, use the Authorize.Net dashboard tools.',
                        'give-authorize'
                    ),
                    $donation->id
                ),
            ]);

            throw new $exception($exception->getMessage());
        }
    }

    /**
     * @since 2.0.0
     *
     * @return void
     *
     * @throws PaymentGatewayException|InvalidCredentialsException
     */
    protected function refundTransaction(Donation $donation)
    {
        (new RefundTransaction())($donation);
    }

    /**
     * @since 2.0.0
     */
    public function addOptRefundCheckbox(int $donationId)
    {
        $donation = Donation::find($donationId);
        if ($donation->gatewayId === self::id()) {
            ?>
            <div id="give-authorize-opt-refund-wrap"
                 class="give-authorize-opt-refund give-admin-box-inside give-hidden">
                <p>
                    <input type="checkbox" id="give-authorize-opt-refund" name="give_authorize_opt_refund" value="1" />
                    <label for="give-authorize-opt-refund">
                        <?php
                        esc_html_e('Refund the donation at Authorize.Net?', 'give-authorize'); ?>
                    </label>
                </p>
            </div>
            <script>
                var donationStatus = document.getElementById('give-payment-status');

                if (!!donationStatus) {
                    donationStatus.addEventListener('change', function (event) {
                        var authorizeCheckbox = document.getElementById('give-authorize-opt-refund');

                        if (null === authorizeCheckbox) {
                            return;
                        }

                        authorizeCheckbox.checked = false;

                        if ('refunded' === event.target.value) {
                            document.getElementById('give-authorize-opt-refund-wrap').style.display = 'block';
                        } else {
                            document.getElementById('give-authorize-opt-refund-wrap').style.display = 'none';
                        }
                    });
                }
            </script>
            <?php
        }
    }

    /**
     * @since 2.0.0
     */
    public function refundDonationAdapter(int $donationId, string $newStatus, string $oldStatus)
    {
        $authorizeOptRefund = ! empty($_POST['give_authorize_opt_refund']) ? give_clean($_POST['give_authorize_opt_refund']) : '';
        $canProcessRefund = ! empty($authorizeOptRefund) ? $authorizeOptRefund : false;

        // Only move forward if refund requested.
        if ( ! $canProcessRefund) {
            return;
        }

        $donation = Donation::find($donationId);
        if ($donation->gatewayId === self::id() &&
            'refunded' === $newStatus &&
            'refunded' !== $oldStatus) {
            $this->refundDonation($donation);
        }
    }
}
