<?php

namespace GiveAuthorizeNet\Gateway\Traits;

use Give\Donations\Models\Donation;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\Exceptions\Primitives\Exception;
use Give\Framework\PaymentGateways\Commands\PaymentComplete;
use Give\Framework\PaymentGateways\Commands\PaymentProcessing;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use GiveAuthorizeNet\Actions\CreateAcceptPaymentTransaction;
use GiveAuthorizeNet\DataTransferObjects\AuthorizeTransactionResponse;
use GiveAuthorizeNet\Exceptions\InvalidCredentialsException;

/**
 * @since 2.0.0
 */
trait CreatePaymentGateway
{
    /**
     * @since 2.0.0
     *
     * @inheritDoc
     *
     * @return PaymentComplete|PaymentProcessing
     * @throws PaymentGatewayException
     */
    public function createPayment(Donation $donation, $gatewayData)
    {
        try {
            $authorizeData = $gatewayData['authorizeGatewayData'];
            $response = $this->createAcceptPaymentTransaction($donation, $authorizeData);

            if ($response->responseCode->isDeclined()) {
                $donation->status = DonationStatus::FAILED();
                $donation->save();
                throw new PaymentGatewayException(
                    sprintf(
                        __(
                            '[[Authorize.Net] The transaction for donation %s was declined.',
                            'give-authorize'
                        ),
                        $donation->id
                    )
                );
            }

            if (!$response->transId || $response->responseCode->isError()) {
                $donation->status = DonationStatus::FAILED();
                $donation->save();
                throw new PaymentGatewayException(
                    sprintf(
                        __(
                            '[[Authorize.Net] There was an error processing donation %s.',
                            'give-authorize'
                        ),
                        $donation->id
                    )
                );
            }

            $donation->gatewayTransactionId = $response->transId;
            $donation->save();
        } catch (Exception $e) {
            throw new PaymentGatewayException($e->getMessage());
        }

        if ($response->responseCode->isReview()) {
            return new PaymentProcessing($donation->gatewayTransactionId);
        }

        return new PaymentComplete($donation->gatewayTransactionId);
    }

    /**
     * @since 2.0.0
     *
     * @throws PaymentGatewayException
     * @throws InvalidCredentialsException
     */
    protected function createAcceptPaymentTransaction(
        $donation,
        $authorizeData
    ): AuthorizeTransactionResponse {
        return (new CreateAcceptPaymentTransaction())($donation, $authorizeData);
    }
}
