import {useEffect, useState} from 'react';
import {PaymentInputsWrapper, usePaymentInputs} from 'react-payment-inputs';
import images from 'react-payment-inputs/images';
import {__} from '@wordpress/i18n';
import {Gateway} from './Interfaces';

type AuthorizeGatewaySettings = {
    acceptJsUrl: string;
    clientPublicKey: string;
    apiLoginId: string;
};

type AuthorizeCardInfo = {
    cardNumber: string;
    cardCode: string;
    month: string;
    year: string;
};

let settings: AuthorizeGatewaySettings;

let cardData: AuthorizeCardInfo = {
    cardNumber: '',
    month: '',
    year: '',
    cardCode: '',
};

let acceptJsDataDescriptor = '';
let acceptJsDataValue = '';

const creditCardGateway: Gateway = {
    id: 'authorize',
    initialize() {
        settings = this.settings;
    },
    beforeCreatePayment: async function (values: FormData): Promise<object> {
        const authData = {
            apiLoginID: settings.apiLoginId,
            clientKey: settings.clientPublicKey,
        };

        const secureData = {
            authData: authData,
            cardData: cardData,
        };

        await new Promise(function (resolve) {
            // @ts-ignore
            Accept.dispatchData(secureData, function (response) {
                let errorMessages = '';
                try {
                    if (response.messages.resultCode === 'Error') {
                        let i = 0;
                        while (i < response.messages.message.length) {
                            errorMessages += `${response.messages.message[i].code} : ${response.messages.message[i].text} \r\n`;
                            i = i + 1;
                        }
                    } else {
                        acceptJsDataDescriptor = response.opaqueData.dataDescriptor
                            ? response.opaqueData.dataDescriptor
                            : '';
                        acceptJsDataValue = response.opaqueData.dataValue ? response.opaqueData.dataValue : '';
                    }
                } catch (error) {
                    throw new Error(error);
                }

                if (errorMessages != '') {
                    throw new Error(errorMessages);
                }

                resolve('Accept.dispatchData');
            });
        });

        return {
            give_authorize_data_descriptor: acceptJsDataDescriptor,
            give_authorize_data_value: acceptJsDataValue,
        };
    },
    Fields() {
        // @ts-ignore
        const {useWatch} = window.givewp.form.hooks;
        const gatewayId = useWatch({name: 'gatewayId'});
        const isCurrentGatewayActive = gatewayId === creditCardGateway.id;

        if (!isCurrentGatewayActive) {
            return;
        }

        const script = document.createElement('script');
        script.src = settings.acceptJsUrl;

        document.body.appendChild(script);

        return <CreditCardFields />;
    },
};

const CreditCardFields = () => {
    const [cardNumber, setCardNumber] = useState('');
    const [cardExpiryDate, setCardExpiryDate] = useState('');
    const [cardCVC, setCardCVC] = useState('');
    const cardFullExpiryYear = cardExpiryDate.substr(5)
        ? new Date().getFullYear().toString().substr(0, 2) + cardExpiryDate.substr(5)
        : '';

    useEffect(() => {
        cardData.cardNumber = cardNumber.replace(/\s+/g, '');
        cardData.month = cardExpiryDate.substr(0, 2);
        cardData.year = cardFullExpiryYear.slice(-2);
        cardData.cardCode = cardCVC;
    }, [cardNumber, cardExpiryDate, cardFullExpiryYear, cardCVC]);

    const {wrapperProps, getCardImageProps, getCardNumberProps, getExpiryDateProps, getCVCProps} = usePaymentInputs();
    const accentColor = 'var(--givewp-primary-color)';

    const inputProps = {
        ...wrapperProps,
        styles: {
            fieldWrapper: {
                base: {
                    width: '100%',
                },
            },
            inputWrapper: {
                base: {
                    display: 'flex',
                    alignItems: 'center',
                    marginTop: '8px',
                    border: '1px solid #b8b8b8',
                    overflow: 'hidden',
                    padding: '4px 14px',
                    boxShadow: `0 0 0 0 ${accentColor}`,
                    transition: 'box-shadow 0.1s ease',
                    borderRadius: '4px',
                },
                errored: {
                    border: '1px solid #b8b8b8',
                    boxShadow: '0 0 0 1px #c9444d',
                },
                focused: {
                    border: '1px solid #b8b8b8',
                    boxShadow: `0 0 0 1px ${accentColor}`,
                },
            },
            input: {
                base: {
                    fontSize: '14px',
                    fontFamily: 'Montserrat, Arial, Helvetica, sans-serif',
                    fontWeight: '500',
                    color: '#828382',
                    lineHeight: '1.2',
                    boxShadow: 'unset',
                },
                cardNumber: {
                    flex: '1',
                },
            },
        },
    };

    const requiredAsterisk = (
        <span className="givewp-field-required" title={__('Field Required', 'give-authorize')}>
            *
        </span>
    );

    return (
        <div style={{marginTop: '10px'}}>
            <label style={{position: 'initial'}}>
                {__('Credit Card Info', 'give-authorize')} {requiredAsterisk}
                <PaymentInputsWrapper {...inputProps}>
                    <svg {...getCardImageProps({images})} />
                    <input
                        {...getCardNumberProps({
                            onChange: (e) => setCardNumber(e.target.value),
                            value: cardNumber,
                        })}
                    />
                    <input
                        {...getExpiryDateProps({
                            onChange: (e) => setCardExpiryDate(e.target.value),
                            value: cardExpiryDate,
                        })}
                    />
                    <input {...getCVCProps({onChange: (e) => setCardCVC(e.target.value), value: cardCVC})} />
                </PaymentInputsWrapper>
            </label>
        </div>
    );
};

// @ts-ignore
window.givewp.gateways.register(creditCardGateway);
