<?php

namespace GiveAuthorizeNet\Gateway;

use GiveAuthorizeNet\Actions\CreateMerchantPublicClientKey;
use GiveAuthorizeNet\DataTransferObjects\ApiAccessData;
use GiveAuthorizeNet\Exceptions\InvalidCredentialsException;

/**
 * Accept.js is a JavaScript library for sending secure payment data directly to Authorize.net. Accept.js captures
 * the payment data and submits it directly to us, in exchange for a one-time-use token, or payment nonce. You can
 * use this payment nonce in the place of payment data in a follow-on createTransactionRequest API call.
 *
 * @see https://developer.authorize.net/api/reference/features/acceptjs.html
 *
 * @since 2.0.0
 */
class AcceptJs
{
    /**
     * @since 2.0.0
     *
     * @return void
     */
    public function addHiddenFields( int $formId ) {
        ?>
        <input type="hidden" id="give_authorize_data_descriptor" name="give_authorize_data_descriptor" value="" />
        <input type="hidden" id="give_authorize_data_value" name="give_authorize_data_value" value="" />
        <?php
    }

    /**
     * @since 3.0
     *
     * @throws InvalidCredentialsException
     */
    public static function getApiAccessData(): array {
        $apiAccessData = ApiAccessData::fromOptions();

        if ( give_is_test_mode() ) {
            $acceptJsUrl     = 'https://jstest.authorize.net/v1/Accept.js';
            $clientPublicKey = ! empty( $apiAccessData->sandboxClientPublicKey ) ? $apiAccessData->sandboxClientPublicKey : ( new CreateMerchantPublicClientKey() )();
            $apiLoginId      = $apiAccessData->sandboxLoginId;
        } else {
            $acceptJsUrl     = 'https://js.authorize.net/v1/Accept.js';
            $clientPublicKey = ! empty( $apiAccessData->liveClientPublicKey ) ? $apiAccessData->liveClientPublicKey : ( new CreateMerchantPublicClientKey() )();
            $apiLoginId      = $apiAccessData->liveLoginId;
        }

        return compact( 'acceptJsUrl', 'clientPublicKey', 'apiLoginId' );
    }

    /**
     * @since 3.0 Move logic to retrieve API access data to getApiAccessData method
     * @since      2.0.4 Display Accept.js errors related to Credit Card fields on the form
     * @since      2.0.1 Use unique selectors by form for the submit button and the "data descriptor" and "data value" hidden fields
     * @since      2.0.0
     *
     * @return void
     * @throws InvalidCredentialsException
     */
    public static function loadScript( int $formId, $creditCard = true ) {
        extract( self::getApiAccessData(), EXTR_PREFIX_SAME, "wddx" );

        ?>
        <script type="text/javascript"
                src="<?php
                echo $acceptJsUrl ?>"
                charset="utf-8">
        </script>
        <script type="text/javascript">

            function readyHandler() {

                let form = document.querySelector(".give-form-<?php echo $formId ?>");
                if (!form) form = document.querySelector(".give-recurring-form");
                let submitButton = document.querySelector("#give-recurring-update-submit");
                let submitButtonDefaultText = '';
                let loadingAnimation = '';

                if (form) {
                    if (!submitButton) submitButton = form.querySelector("#give-purchase-button");
                    submitButtonDefaultText = submitButton.value;
                    loadingAnimation = form.querySelector('input[type="submit"].give-submit + .give-loading-animation');
                    submitButton.addEventListener("click", sendPaymentDataToAnet);

                    /**
                     * Prevent checkValidity() from returning a wrong false result when setCustomValidity() is set
                     *
                     * @see https://stackoverflow.com/a/70746845
                     * @see https://www.jabcreations.com/blog/fix-checkvalidity-returning-false-for-valid-form-field
                     */
                    for (let i = 0; i < form.elements.length; i++) {
                        form.elements[i].addEventListener('change', function (event) {
                            event.target.setCustomValidity('');
                        });
                    }
                }

                async function sendPaymentDataToAnet(e) {
                    e.stopImmediatePropagation();

                    if (form.checkValidity() === false) {
                        return;
                    }

                    e.preventDefault();

                    if (give_global_vars.purchase_loading) submitButton.value = give_global_vars.purchase_loading;
                    submitButton.disabled = true;

                    if (loadingAnimation) loadingAnimation.style.display = "block";

                    authData = {};
                    authData.clientKey = "<?php echo $clientPublicKey ?>";
                    authData.apiLoginID = "<?php echo $apiLoginId ?>";

                    const secureData = {};
                    secureData.authData = authData;

                    <?php if ($creditCard): ?>
                    const cardData = {};
                    cardData.cardNumber = document.getElementById("card_number-<?php echo $formId ?>").value.replace(/\s/g, ''); //2223000010309711
                    cardData.month = document.getElementById("card_exp_month-<?php echo $formId ?>").value; //12
                    cardData.year = document.getElementById("card_exp_year-<?php echo $formId ?>").value.slice(-2); //34
                    cardData.cardCode = document.getElementById("card_cvc-<?php echo $formId ?>").value; //147
                    secureData.cardData = cardData;
                    <?php else: ?> // If using banking information instead of card information, build a bankData object instead of a cardData object.
                    const bankData = {};
                    bankData.routingNumber = document.getElementById("routing-number-<?php echo $formId ?>").value; //062204569
                    bankData.accountNumber = document.getElementById("account-number-<?php echo $formId ?>").value; //123456789
                    bankData.nameOnAccount = document.getElementById("name-on-account-<?php echo $formId ?>").value; //John Doe
                    bankData.accountType = document.getElementById("account-type-<?php echo $formId ?>").value; //checking
                    secureData.bankData = bankData;
                    <?php endif; ?>

                    await dispatchData(secureData);
                }

                function dispatchData(secureData) {
                    return new Promise(function (resolve) {
                        Accept.dispatchData(secureData, responseHandler);
                        resolve("Accept.dispatchData");
                    });
                }

                function responseHandler(response) {
                    try {
                        if (response.messages.resultCode === "Error") {

                            submitButton.disabled = false;
                            submitButton.value = submitButtonDefaultText;
                            if (loadingAnimation) loadingAnimation.style.display = "none";

                            let i = 0;
                            while (i < response.messages.message.length) {
                                console.log(
                                    response.messages.message[i].code + ": " +
                                    response.messages.message[i].text,
                                );

                                /**
                                 * @see https://developer.authorize.net/api/reference/features/acceptjs.html#Appendix_Error_Codes
                                 */
                                if ('E_WC_05' === response.messages.message[i].code) {
                                    const cardNumberInput = document.getElementById("card_number-<?php echo $formId ?>");
                                    cardNumberInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if ('E_WC_06' === response.messages.message[i].code ||
                                    'E_WC_07' === response.messages.message[i].code ||
                                    'E_WC_08' === response.messages.message[i].code) {
                                    const cardExpiryInput = document.getElementById("card_expiry-<?php echo $formId ?>");
                                    cardExpiryInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if ('E_WC_15' === response.messages.message[i].code) {
                                    const cardCvcInput = document.getElementById("card_cvc-<?php echo $formId ?>");
                                    cardCvcInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if ('E_WC_24' === response.messages.message[i].code) {
                                    const accountNumberInput = document.getElementById("account-number-<?php echo $formId ?>");
                                    accountNumberInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if ('E_WC_25' === response.messages.message[i].code) {
                                    const routingNumberInput = document.getElementById("routing-number-<?php echo $formId ?>");
                                    routingNumberInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if ('E_WC_26' === response.messages.message[i].code) {
                                    const nameOnAccountInput = document.getElementById("name-on-account-<?php echo $formId ?>");
                                    nameOnAccountInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if ('E_WC_27' === response.messages.message[i].code) {
                                    const accountTypeInput = document.getElementById("account-type-<?php echo $formId ?>");
                                    accountTypeInput.setCustomValidity(response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }
                                if (response.messages.message[i].code) {
                                    submitButton.setCustomValidity('Authorize.Net Error: ' + response.messages.message[i].code + ' - ' + response.messages.message[i].text);
                                    submitButton.click();
                                    break;
                                }

                                i = i + 1;
                            }
                        } else {
                            paymentFormUpdate(response.opaqueData);
                        }
                    } catch (error) {
                        console.log(error);
                    }
                }

                function paymentFormUpdate(opaqueData) {
                    form.querySelector("#give_authorize_data_descriptor").value = opaqueData.dataDescriptor;
                    form.querySelector("#give_authorize_data_value").value = opaqueData.dataValue;
                    clearSensitiveData();
                    form.submit();
                }

                function clearSensitiveData() {
                    <?php if ($creditCard): ?>
                    document.getElementById("card_number-<?php echo $formId ?>").value = '0000000000000000';
                    document.getElementById("card_expiry-<?php echo $formId ?>").value = '00 / 00';
                    document.getElementById("card_exp_month-<?php echo $formId ?>").value = '00';
                    document.getElementById("card_exp_year-<?php echo $formId ?>").value = '00';
                    document.getElementById("card_name-<?php echo $formId ?>").value = '0000000000000000';
                    document.getElementById("card_cvc-<?php echo $formId ?>").value = '000';
                    <?php else: ?>
                    document.getElementById("routing-number-<?php echo $formId ?>").value = '000000000';
                    document.getElementById("account-number-<?php echo $formId ?>").value = '000000000';
                    document.getElementById("name-on-account-<?php echo $formId ?>").value = '000000000';
                    <?php endif; ?>
                }
            }

            // Check if the DOMContentLoaded has already been completed
            if (document.readyState !== 'loading') {
                readyHandler();
            } else {
                document.addEventListener('DOMContentLoaded', readyHandler);
            }
        </script>
        <?php
    }
}
