<?php

namespace GiveAuthorizeNet\DataTransferObjects;

use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;

/**
 * A structured representation of the Authorize.net gateway data sent by the Accept.js library
 *
 * @see https://developer.authorize.net/api/reference/features/acceptjs.html
 *
 * @since 2.0.0
 */
class AuthorizeGatewayData
{
    /**
     *
     * Identifies the type or source of opaque payment data.
     * For Accept transactions, the value must be COMMON.ACCEPT.INAPP.PAYMENT
     *
     * @var string
     */
    public $dataDescriptor;

    /**
     *
     * The unmodified token or encrypted data BLOB that represents the payment data.
     *
     * @var string
     */
    public $dataValue;

    /**
     * @since 2.0.0
     *
     * @param array $request {give_authorize_data_descriptor: string, give_authorize_data_value: string}
     *
     * @return AuthorizeGatewayData
     *
     * @throws PaymentGatewayException
     */
    public static function fromRequest(array $request): AuthorizeGatewayData
    {
        $self = new self();

        $self->dataDescriptor = $request['give_authorize_data_descriptor'] ?? '';
        $self->dataValue = $request['give_authorize_data_value'] ?? '';

        if (empty($self->dataDescriptor) || empty($self->dataValue)) {
            throw new PaymentGatewayException(
                __('[Authorize.Net] The "data descriptor" or "data value" is missing.', 'give-authorize'));
        }

        return $self;
    }
}
