<?php

namespace GiveAuthorizeNet\Actions;

use Give\Donations\Models\Donation;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give_Authorize;
use GiveAuthorizeNet\Exceptions\InvalidCredentialsException;
use GiveAuthorizeNet\ValueObjects\AuthorizeApiResultCode;
use net\authorize\api\contract\v1 as AnetAPI;
use net\authorize\api\contract\v1\ANetApiResponseType;
use net\authorize\api\contract\v1\TransactionResponseType;
use net\authorize\api\controller as AnetController;

/**
 * @since 2.0.0
 */
class RefundTransaction
{
    /**
     * This transaction type is used to refund a customer for a transaction that was successfully
     * settled through the payment gateway. Note that credit card information and bank account
     * information are mutually exclusive, so you should not submit both.
     *
     * @see https://developer.authorize.net/api/reference/index.html#payment-transactions-refund-a-transaction
     *
     * @since 2.0.0
     * @throws PaymentGatewayException
     * @throws InvalidCredentialsException
     */
    public function __invoke(Donation $donation): bool
    {
        $gatewayTransactionId = $donation->gatewayTransactionId;

        if ( ! $gatewayTransactionId) {
            throw new PaymentGatewayException(__('[Authorize.Net] The transaction id was not provided during an attempted refund. Check the donation details and ensure that the transaction ID is present and matches the one from the gateway.',
                'give-authorize'));
        }

        $transactionDetails = (new GetTransactionDetails())($gatewayTransactionId);

        if ( ! $transactionDetails) {
            throw new PaymentGatewayException(__('[Authorize.Net] During an attempted refund, transaction details were not returned from the gateway. Ensure that the transaction ID matches on the gateway side.',
                'give-authorize'));
        }

        $merchantAuthentication = (new CreateMerchantAuthentication())();

        // Set the transaction's refId
        $refId = 'ref' . time();

        $payment = new AnetAPI\PaymentType();

        /**
         * Refunds submitted via API are only available to eCheck.Net transactions submitted with the CCD or PPD eCheck Type.
         *
         * @see https://support.authorize.net/knowledgebase/Knowledgearticle/?code=000001388
         */
        if ('authorize_echeck' === $donation->gatewayId) {
            // Create the payment data for a Bank Account
            $bankAccount = new AnetAPI\BankAccountType();
            $bankAccount->setAccountType($transactionDetails->getPayment()->getBankAccount()->getAccountType());
            $bankAccount->setEcheckType($transactionDetails->getPayment()->getBankAccount()->getEcheckType());
            $bankAccount->setRoutingNumber($transactionDetails->getPayment()->getBankAccount()->getRoutingNumber());
            $bankAccount->setAccountNumber($transactionDetails->getPayment()->getBankAccount()->getAccountNumber());
            $bankAccount->setNameOnAccount($transactionDetails->getPayment()->getBankAccount()->getNameOnAccount());
            $bankAccount->setBankName($transactionDetails->getPayment()->getBankAccount()->getBankName());
            $payment->setBankAccount($bankAccount);
        } else {
            // Create the payment data for a credit card
            $creditCard = new AnetAPI\CreditCardType();
            $creditCard->setCardNumber($transactionDetails->getPayment()->getCreditCard()->getCardNumber());
            $creditCard->setExpirationDate($transactionDetails->getPayment()->getCreditCard()->getExpirationDate());
            $payment->setCreditCard($creditCard);
        }

        //create a transaction
        $transactionRequest = new AnetAPI\TransactionRequestType();
        $transactionRequest->setTransactionType("refundTransaction");
        $transactionRequest->setAmount($donation->amount->formatToDecimal());
        $transactionRequest->setPayment($payment);
        $transactionRequest->setRefTransId($donation->gatewayTransactionId);


        $request = new AnetAPI\CreateTransactionRequest();
        $request->setMerchantAuthentication($merchantAuthentication);
        $request->setRefId($refId);
        $request->setTransactionRequest($transactionRequest);
        $controller = new AnetController\CreateTransactionController($request);

        $apiResponse = $controller->executeWithApiResponse(Give_Authorize::get_instance()->getApiEnv());

        if ($apiResponse === null) {
            $this->logErrorNoResponse($donation);

            throw new PaymentGatewayException(
                sprintf(
                    '[Authorize.Net] The gateway did not respond to an attempt to refund donation %s. Check to confirm whether the transaction was refunded on the gateway side.',
                    $donation->id
                )
            );
        }

        $transactionResponse = $apiResponse->getTransactionResponse();

        if ($transactionResponse === null) {
            $this->logErrorInvalidTransaction($donation);

            throw new PaymentGatewayException(
                sprintf(
                    '[Authorize.Net] The gateway response to an attempted refund for donation %s was null. Check to confirm whether the transaction was refunded on the gateway side.',
                    $donation->id
                )
            );
        }

        if ($apiResponse->getMessages()->getResultCode() !== AuthorizeApiResultCode::OK) {
            $this->logErrorTransactionFailed($donation, $transactionResponse, $apiResponse);

            throw new PaymentGatewayException(
                sprintf(
                    '[Authorize.Net] Refund failed for donation %s. The refund can be initiated on the gateway side, or try again here.',
                    $donation->id
                )
            );
        }

        $this->logSuccessfulTransaction($donation, $transactionResponse);

        return true;
    }

    /**
     * @since 2.0.0
     */
    private function logErrorNoResponse(Donation $donation)
    {
        PaymentGatewayLog::error(
            sprintf('[Authorize.Net] The gateway did not respond to an attempt to refund donation %s. Check to confirm whether the transaction was refunded on the gateway side.',
                $donation->id),
            [
                'Payment Gateway' => $donation->gateway()->getId(),
                'Donation' => $donation->id,
            ]
        );
    }

    /**
     * @since 2.0.0
     */
    private function logErrorInvalidTransaction(Donation $donation)
    {
        PaymentGatewayLog::error(
            sprintf('[Authorize.Net] The gateway response to an attempted refund for donation %s was null. Check to confirm whether the transaction was refunded on the gateway side.',
                $donation->id),
            [
                'Payment Gateway' => $donation->gateway()->getId(),
                'Donation' => $donation->id,
            ]
        );
    }

    /**
     * @since 2.0.0
     */
    private function logErrorTransactionFailed(
        Donation $donation,
        TransactionResponseType $transactionResponse,
        AnetApiResponseType $apiResponse
    ) {
        if ($transactionResponse->getErrors() != null) {
            $errorCode = $transactionResponse->getErrors()[0]->getErrorCode();
            $errorMessage = $transactionResponse->getErrors()[0]->getErrorText();
        } else {
            $errorCode = $apiResponse->getMessages()->getMessage()[0]->getCode();
            $errorMessage = $apiResponse->getMessages()->getMessage()[0]->getText();
        }

        PaymentGatewayLog::error(
            sprintf('[Authorize.Net] Refund failed for donation %s. The refund can be initiated on the gateway side, or try again here.',
                $donation->id),
            [
                'Error Code' => $errorCode,
                'Error Message' => $errorMessage,
            ]
        );
    }

    /**
     * @since 2.0.0
     */
    private function logSuccessfulTransaction(Donation $donation, TransactionResponseType $transactionResponse)
    {
        PaymentGatewayLog::success(
            sprintf('[Authorize.Net] Donation %s was successfully refunded.', $donation->id),
            [
                'Payment Gateway' => $donation->gateway()->getId(),
                'Donation' => $donation->id,
                'Transaction ID' => $transactionResponse->getTransId(),
                'Transaction Response Code' => $transactionResponse->getResponseCode(),
                'Message Code' => $transactionResponse->getMessages() ? $transactionResponse->getMessages()[0]->getCode() : 'null',
                'Auth Code' => $transactionResponse->getAuthCode(),
                'Description' => $transactionResponse->getMessages() ? $transactionResponse->getMessages()[0]->getDescription() : 'null',
            ]
        );
    }
}
