<?php

namespace GiveAuthorizeNet\Actions;

use Give\Donations\Models\Donation;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give_Authorize;
use GiveAuthorizeNet\DataTransferObjects\AuthorizeGatewayData;
use GiveAuthorizeNet\DataTransferObjects\AuthorizeTransactionResponse;
use GiveAuthorizeNet\Exceptions\InvalidCredentialsException;
use GiveAuthorizeNet\ValueObjects\AuthorizeApiResponseCode;
use GiveAuthorizeNet\ValueObjects\AuthorizeApiResultCode;
use net\authorize\api\contract\v1 as AnetAPI;
use net\authorize\api\contract\v1\AnetApiResponseType;
use net\authorize\api\contract\v1\TransactionResponseType;
use net\authorize\api\controller as AnetController;

/**
 * @since 2.0.0
 */
class CreateAcceptPaymentTransaction
{
    /**
     * Use this function to create an Authorize.net payment transaction
     * request, using the Accept Payment nonce in place of card data.
     *
     * @see        https://developer.authorize.net/api/reference/index.html#accept-suite-create-an-accept-payment-transaction
     *
     * @since 3.0.1 Set currency code
     * @since 2.0.3 Add givewp_authorize_one_time_payment_description filter
     * @since      2.0.2 Use setCustomerIP() to prevent the Authorize.Net fraud filters in sites that receive too many donations.
     * @since      2.0.0
     *
     * @throws InvalidCredentialsException|PaymentGatewayException
     */
    public function __invoke(
        Donation $donation,
        AuthorizeGatewayData $authorizeData
    ): AuthorizeTransactionResponse {
        $merchantAuthentication = (new CreateMerchantAuthentication())();

        // Set the transaction's refId
        $refId = 'ref' . time();

        // Create the payment object for a payment nonce
        $opaqueData = new AnetAPI\OpaqueDataType();
        $opaqueData->setDataDescriptor($authorizeData->dataDescriptor);
        $opaqueData->setDataValue($authorizeData->dataValue);

        // Add the payment data to a paymentType object
        $paymentOne = new AnetAPI\PaymentType();
        $paymentOne->setOpaqueData($opaqueData);

        // Create order information
        $order = new AnetAPI\OrderType();
        $order->setInvoiceNumber($donation->id);
        $description = apply_filters('givewp_authorize_one_time_payment_description', 'GiveWP Donation', $donation);
        $order->setDescription($description);

        // Set the customer's Bill To address
        $customerAddress = new AnetAPI\CustomerAddressType();
        $customerAddress->setFirstName($donation->donor->firstName);
        $customerAddress->setLastName($donation->donor->lastName);
        $customerAddress->setCompany($donation->company);
        $customerAddress->setAddress($donation->billingAddress->address1 . ' ' . $donation->billingAddress->address2);
        $customerAddress->setCity($donation->billingAddress->city);
        $customerAddress->setState($donation->billingAddress->state);
        $customerAddress->setZip($donation->billingAddress->zip);
        $customerAddress->setCountry($donation->billingAddress->country);

        // Set the customer's identifying information
        $customerData = new AnetAPI\CustomerDataType();
        $customerData->setType("individual"); //or business
        $customerData->setId($donation->donorId);
        $customerData->setEmail($donation->donor->email);

        // Create a TransactionRequestType object and add the previous objects to it
        $transactionRequestType = new AnetAPI\TransactionRequestType();
        $transactionRequestType->setCustomerIP($donation->donorIp);
        $transactionRequestType->setTransactionType("authCaptureTransaction");
        $transactionRequestType->setAmount($donation->amount->formatToDecimal());
        $transactionRequestType->setCurrencyCode($donation->amount->getCurrency()->getCode());
        $transactionRequestType->setOrder($order);
        $transactionRequestType->setPayment($paymentOne);
        $transactionRequestType->setBillTo($customerAddress);
        $transactionRequestType->setCustomer($customerData);

        // Assemble the complete transaction request
        $request = new AnetAPI\CreateTransactionRequest();
        $request->setMerchantAuthentication($merchantAuthentication);
        $request->setRefId($refId);
        $request->setTransactionRequest($transactionRequestType);

        // Create the controller and get the response
        $controller = new AnetController\CreateTransactionController($request);

        $apiResponse = $controller->executeWithApiResponse(Give_Authorize::get_instance()->getApiEnv());

        if ($apiResponse === null) {
            $this->logErrorNoResponse($donation);
        }

        $transactionResponse = $apiResponse->getTransactionResponse();

        if ($transactionResponse === null) {
            $this->logErrorInvalidTransaction($donation);
        }

        if ($apiResponse->getMessages()->getResultCode() !== AuthorizeApiResultCode::OK) {
            $this->logErrorTransactionFailed($donation, $transactionResponse, $apiResponse);
        }

        $this->logSuccessfulTransaction($donation, $transactionResponse);

        return AuthorizeTransactionResponse::fromArray([
            'responseCode' => new AuthorizeApiResponseCode($transactionResponse->getResponseCode()),
            'transId' => $transactionResponse->getTransId(),
        ]);
    }

    /**
     * @since 2.0.0
     * @throws PaymentGatewayException
     */
    private function logErrorNoResponse(Donation $donation)
    {
        PaymentGatewayLog::error(
            sprintf('[Authorize.Net] No response from the API. We\'re unable to contact the payment gateway to complete donation %s.',
                $donation->id),
            [
                'Payment Gateway' => $donation->gateway()->getId(),
                'Donation' => $donation->id,
            ]
        );

        throw new PaymentGatewayException(__('[Authorize.Net] No response from the API.', 'give-authorize'));
    }

    /**
     * @since 2.0.0
     * @throws PaymentGatewayException
     */
    private function logErrorInvalidTransaction(Donation $donation)
    {
        PaymentGatewayLog::error(
            sprintf('[Authorize.Net] Invalid transaction response. Unable to complete donation %s.', $donation->id),
            [
                'Payment Gateway' => $donation->gateway()->getId(),
                'Donation' => $donation->id,
            ]
        );

        throw new PaymentGatewayException(__('[Authorize.Net] Invalid response from the API.', 'give-authorize'));
    }

    /**
     * @since 3.0.1 Add custom message for error code 39
     * @since 2.0.0
     * @throws PaymentGatewayException
     */
    private function logErrorTransactionFailed(
        Donation $donation,
        TransactionResponseType $transactionResponse,
        AnetApiResponseType $apiResponse
    ) {
        if ($transactionResponse->getErrors() != null) {
            $errorCode = $transactionResponse->getErrors()[0]->getErrorCode();
            $errorMessage = $transactionResponse->getErrors()[0]->getErrorText();
        } else {
            $errorCode = $apiResponse->getMessages()->getMessage()[0]->getCode();
            $errorMessage = $apiResponse->getMessages()->getMessage()[0]->getText();
        }

        PaymentGatewayLog::error(
            sprintf('[Authorize.net] Transaction Failed for donation %s', $donation->id),
            [
                'Error Code' => $errorCode,
                'Error Message' => $errorMessage,
            ]
        );

        /**
         * There are two possible causes of this error:
         *
         * 1) This error may occur if you use the field x_currency_code in your scripting, and you are setting
         * it to a currency code other than what your account is set up for. Only one currency can be set for
         * one account. At this time, Authorize.net only supports the following currencies: AUD, CAD, CHF, DKK,
         * EUR, GBP, JPY, NOK, NZD, SEK, USD, ZAR.
         *
         * 2) This error may occur when an Authorize.net account is created without a valid Currency ID. In this
         * situation, processing transactions is not possible through the API or through the Virtual Point of Sale,
         * regardless of the currency you choose.
         *
         * @see https://support.authorize.net/knowledgebase/Knowledgearticle/?code=000001351
         */
        if (39 === (int)$errorCode) {
            $errorMessage = sprintf(__('The selected payment method does not support %s currency, please select another currency or payment method',
                'give-authorize'), $donation->amount->getCurrency()->getCode());
        } else {
            $errorMessage = sprintf(__('[Authorize.net] Transaction Failed. Error: %s - %s', 'give-authorize'),
                $errorCode, $errorMessage);
        }

        throw new PaymentGatewayException($errorMessage, $errorCode);
    }

    /**
     * @since 2.0.0
     */
    private function logSuccessfulTransaction(Donation $donation, TransactionResponseType $transactionResponse)
    {
        PaymentGatewayLog::success(
            sprintf('[Authorize.net] Transaction Successful for donation %s.', $donation->id),
            [
                'Payment Gateway' => $donation->gateway()->getId(),
                'Donation' => $donation->id,
                'Transaction ID' => $transactionResponse->getTransId(),
                'Transaction Response Code' => $transactionResponse->getResponseCode(),
                'Message Code' => $transactionResponse->getMessages() ? $transactionResponse->getMessages()[0]->getCode() : 'null',
                'Auth Code' => $transactionResponse->getAuthCode(),
                'Description' => $transactionResponse->getMessages() ? $transactionResponse->getMessages()[0]->getDescription() : 'null',
            ]
        );
    }
}
