<?php
/**
 * Give_Authorize_Payments
 *
 * @package     Give
 * @since       1.1
 * @license     https://opensource.org/licenses/gpl-license GNU Public License
 * @copyright   Copyright (c) 2017, GiveWP
 */

// Exit if accessed directly

if ( ! defined('ABSPATH')) {
    exit;
}

/**
 * Give_Authorize_Payments
 */
class Give_Authorize_Payments
{

    /**
     * @var string
     */
    var $live_login_id;

    /**
     * @var string
     */
    var $live_transaction_key;

    /**
     * Whether the webhooks have been setup.
     *
     * @var bool
     */
    var $live_webhooks_setup;

    /**
     * @var string
     */
    var $sandbox_login_id;

    /**
     * @var string
     */
    var $sandbox_transaction_key;

    /**
     * Whether the webhooks have been setup.
     *
     * @var bool
     */
    var $sandbox_webhooks_setup;

    /**
     * Give_Authorize_Payments constructor.
     */
    public function __construct()
    {
        require GIVE_AUTHORIZE_PLUGIN_DIR . '/vendor/autoload.php';

		$this->live_login_id        = give_get_option( 'give_api_login' );
		$this->live_transaction_key = give_get_option( 'give_transaction_key' );
		$this->live_webhooks_setup  = give_get_option( 'give_authorize_live_webhooks_setup' );

		$this->sandbox_login_id        = give_get_option( 'give_authorize_sandbox_api_login' );
		$this->sandbox_transaction_key = give_get_option( 'give_authorize_sandbox_transaction_key' );
		$this->sandbox_webhooks_setup  = give_get_option( 'give_authorize_sandbox_webhooks_setup' );

		$this->hooks();
	}


	/**
	 * Setup hooks.
	 */
	public function hooks() {

        add_filter('give_payment_details_transaction_id-authorize', array( $this, 'link_transaction_id' ), 10, 2 );

		add_filter( 'give_payment_details_transaction_id-authorize_echeck', array(
			$this,
			'link_transaction_id'
		), 10, 2 );
	}

	/**
	 * Sets up the API Request.
	 *
	 * @return  \JohnConde\Authnet\AuthnetJsonResponse object|bool
	 */
	function setup_api_request() {

		$missing_key_msg       = __( 'There was a problem processing this donation. Please contact the site administrator for assistance.', 'give-authorize' );
		$missing_key_admin_msg = __( 'An Authorize.net transaction was attempted without entering the proper API credentials within the plugin settings.', 'give-authorize' );

		try {
			if ( give_is_test_mode() ) {

				// Keys = mucho necessario.
				if ( empty( $this->sandbox_login_id ) || empty( $this->sandbox_transaction_key ) ) {
					give_record_gateway_error( esc_html__( 'Authorize.net Error', 'give-authorize' ), $missing_key_admin_msg );
					give_set_error( 'give-authorize-missing-keys', $missing_key_msg );
					give_send_back_to_checkout( '?payment-mode=authorize' );

					return false;
				}

				$request = \JohnConde\Authnet\AuthnetApiFactory::getJsonApiHandler( $this->sandbox_login_id, $this->sandbox_transaction_key,
					1 );

			} else {

				// We need keys to continue
				if ( empty( $this->live_login_id ) || empty( $this->live_transaction_key ) ) {
					give_record_gateway_error( esc_html__( 'Authorize.net Error', 'give-authorize' ), $missing_key_admin_msg );
					give_set_error( 'give-authorize-missing-keys', $missing_key_msg );
					give_send_back_to_checkout( '?payment-mode=authorize' );

					return false;
				}

				$request = \JohnConde\Authnet\AuthnetApiFactory::getJsonApiHandler( $this->live_login_id, $this->live_transaction_key,
					0 );

			}

			return $request;

		} catch ( \Exception $e ) {

			give_record_gateway_error( esc_html__( 'Authorize.net Error', 'give-authorize' ), $e->getMessage() );
			give_set_error( 'error_id_here', __( 'Missing API Login or Transaction key. Please enter them in the plugin settings.', 'give-authorize' ) );

			return false;

		}// End try().

	}

	/**
	 * Link transaction ID.
	 *
	 * Links the transaction ID in the donation details the ID to the authnet trans search with a tooltip.
	 *
	 * @since 1.3
	 *
	 * @param $transaction_id
	 * @param $payment_id
	 *
	 * @return string
	 */
	public function link_transaction_id( $transaction_id, $payment_id ) {

		if ( intval( $transaction_id ) === intval( $payment_id ) ) {
			return $transaction_id;
		}

		if ( ! empty( $transaction_id ) ) {

			$payment = new Give_Payment( $payment_id );
			$html    = '<a href="%s" target="_blank">' . $transaction_id . '</a> <span class="dashicons dashicons-editor-help"
data-tooltip="' . __( 'This is the Authorize.net transction ID. Click the link to view more details at the gateway.', 'give-authorize' ) . '"></span>';

			// Set appropriate URL.
			$base_url = 'live' === $payment->mode ? 'https://authorize.net/ui/themes/sandbox/merch.aspx?page=search' : 'https://sandbox.authorize.net/ui/themes/sandbox/merch.aspx?page=search';
			$link     = esc_url( $base_url );

			// Setup html.
			$transaction_id = sprintf( $html, $link );
		}

		return $transaction_id;

	}
}
